# -*- encoding: utf-8 -*-
import functools
import os
import sys
import tempfile
from copy import copy
from datetime import datetime, timedelta
from itertools import chain
from optparse import Option, OptionParser, OptionValueError
from past.builtins import reload

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

DESTINATION_TABLE_INDEX = 0
UNPARSED_TABLE_INDEX = 1
NATIONAL_CURRENCIES = {
    'ru': 'RUR',
    'ua': 'UAH',
    'kz': 'KZT',
    'tr': 'TRY',
    'com': 'EUR',
}
CAPITALS_MAP = {
    'ru': 213,     # Москва
    'ua': 143,     # Киев
    'tr': 11508,   # Стамбул
    'kz': 163,     # Астана
    'com': 10376,  # Брюссель
}
FOREIGN_CURRENCY_COEFF = 1.03
MAX_MYSQL_INT_VALUE = 2147483647
MAX_MYSQL_PK_VALUE = int(MAX_MYSQL_INT_VALUE * .9)
JSON_FORMAT_ENCODE_UTF8_TRUE = yt.JsonFormat(attributes={'encode_utf8': True})
JSON_FORMAT_ENCODE_UTF8_FALSE = yt.JsonFormat(attributes={'encode_utf8': False})


def fast_date_convert(pydate):
    """
    2014-12-31
    year, month, day
    """
    return datetime(int(pydate[0:4]), int(pydate[5:7]), int(pydate[8:10])).date()


def parse_routes(routes):
    return {r for r in routes.replace(';', ',').replace('/', ',').split(',')}


def check_number(option, opt, value):
    try:
        days=int(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad number: %r" % (opt, value))

    return days


def test_data(existing_records_count, new_records_count, log=None):
    delta = abs(existing_records_count - new_records_count)

    try:
        percent = int((float(delta) / float(existing_records_count)) * 100)

    except ZeroDivisionError:
        percent = None

    if percent >= 30:
        if log:
            log.error('New/Existing data diff >= 30%%: %s / %s (delta %s) = %s%%' % (
                new_records_count, existing_records_count, delta, percent
            ))

        return False

    elif percent >= 10 and log:
        log.warning('New/Existing data diff >= 10%%: %s / %s (delta %s) = %s%%' % (
            new_records_count, existing_records_count, delta, percent
        ))

    elif log:
        log.info('New/Existing data diff: %s / %s (delta %s) = %s%%' % (
            new_records_count, existing_records_count, delta, percent
        ))

    return True


# Convert station to settlement
def fix_point(point_key, stations_map):
    if point_key.startswith('s'):
        return stations_map.get(point_key)

    return point_key


def get_currency_map():
    from travel.avia.library.python.common.models.currency import Currency
    return {c.code: c.id for c in Currency.objects.all()}


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.conf import settings
    from django.db import connection, transaction
    from django.db.models import Min

    from travel.avia.library.python.avia_data.models.prices import AvgPrice, MinCountryPrice, MinPrice
    from travel.avia.library.python.iata_correction import IATACorrector

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper
    from travel.avia.admin.avia_scripts.fill_livestore_data import run_fill_prices, DEFAULT_NATIONAL_COUNTRIES

    FAKE_EVENTTIME = datetime(2000, 1, 1)

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    currency_map = get_currency_map()

    def store_avg_prices(avg_prices_table, log):
        # Cache table to local file
        with tempfile.NamedTemporaryFile() as f:
            log.info('Read avg price from %s to %s' % (avg_prices_table, f.name))
            prices = {}

            for record in yt.read_table(avg_prices_table, format=JSON_FORMAT_ENCODE_UTF8_FALSE, raw=False):
                direction_parths = record['direction'].split('_')
                from_id = direction_parths[0][1:]
                to_id = direction_parths[1][1:]
                national_version = direction_parths[2]
                direct_flight = direction_parths[3]
                price, currency_code = record['price'].split(' ')
                date_forward = fast_date_convert(record['date_forward'])
                date_backward = fast_date_convert(record['date_backward']) if record['date_backward'] else None

                key = (
                    from_id,
                    to_id,
                    direct_flight,
                    national_version,
                    date_forward.month,
                    date_forward.year,
                    date_backward.month if date_backward else None,
                    date_backward.year if date_backward else None,
                    record['passengers'],
                    currency_code
                )

                if key not in prices:
                    prices[key] = []

                prices[key].append(int(round(float(price))))

            record_count = 0
            for (from_id, to_id, direct_flight, national_version, date_forward_month, date_forward_year,
                 date_backward_month, date_backward_year, passengers, currency_code), prices_list in prices.items():
                average_price = int(round(float(sum(prices_list)) / len(prices_list)))
                currency_id = currency_map.get(currency_code)

                if currency_id:
                    mysql_line = '%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s\n' % (
                        from_id,
                        to_id,
                        national_version,
                        direct_flight,
                        average_price,
                        currency_id,
                        date_forward_month,
                        date_forward_year,
                        date_backward_month,
                        date_backward_year,
                        passengers,
                    )
                    f.writelines(mysql_line)
                    record_count += 1

                else:
                    log.warning('Unknown currency code: %s', currency_code)

            f.flush()
            os.fsync(f.fileno())

            log.info('%s lines flushed', record_count)

            write_avg_price_results(f, record_count, log)

    def write_avg_price_results(f, record_count, log):
        avg_prices_db_count = AvgPrice.objects.all().count()

        if options.skip_check:
            log.info('Skip database check')

        elif not test_data(avg_prices_db_count, record_count, log):
            log.error('Skip database update for avg prices')

            return False

        f.flush()
        os.fsync(f.fileno())

        with transaction.atomic():
            log.info('Delete from avia_avgprice')
            AvgPrice.objects.all().delete()

            log.info('Mysql load data to avia_avgprice')
            connection.cursor().execute("""
            LOAD DATA LOCAL INFILE %s INTO TABLE avia_avgprice
                CHARACTER SET utf8 FIELDS TERMINATED BY '|'
            (
                departure_settlement_id, arrival_settlement_id, national_version, direct_flight, price,currency_id, month_forward, year_forward, @month_backward, @year_backward, passengers
            )
            SET
            month_backward = NULLIF(@month_backward, 'None'),
            year_backward = NULLIF(@year_backward, 'None')
            """, [f.name])

        avg_prices_db_count = AvgPrice.objects.all().count()
        log.info('%s avg prices saved', avg_prices_db_count)

    def store_min_prices(min_prices_table, cursor, log):
        with tempfile.NamedTemporaryFile() as f:
            log.info('Read %s and write to %s' % (min_prices_table, f.name))
            record_count = 0
            for record in yt.read_table(min_prices_table, format=JSON_FORMAT_ENCODE_UTF8_TRUE, raw=False):
                direction_parts = record['direction'].split('_')
                from_id = direction_parts[0][1:]
                to_id = direction_parts[1][1:]
                national_version = direction_parts[2]
                direct_flight = direction_parts[3]
                price, currency_code = record['price'].split(' ')
                currency_id = currency_map.get(currency_code)

                day_of_week = fast_date_convert(record['date_forward']).weekday()

                if currency_id:
                    mysql_line = '%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s\n' % (
                        from_id,
                        to_id,
                        price,
                        currency_id,
                        record['iso_eventtime'],
                        record['date_forward'],
                        national_version,
                        direct_flight,
                        day_of_week,
                        record['date_backward'] if record['date_backward'] else 'None',
                        record['passengers'],
                        record['routes'],
                    )

                    f.writelines(mysql_line)
                    record_count += 1
                else:
                    log.warning('Unknown currency code: %s', currency_code)

            f.flush()
            os.fsync(f.fileno())

            log.info('%s lines flushed', record_count)

            write_min_price_results(f, record_count, cursor, log)

    def check_auto_increment(table, cursor):
        log.info('Check AUTO_INCREMENT')
        cursor.execute(
            'SELECT `AUTO_INCREMENT` FROM `INFORMATION_SCHEMA`.`TABLES`'
            ' WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s', [
                settings.DATABASES['default']['NAME'], '{table}'.format(table=table)
            ]
        )
        return cursor.fetchone()[0]

    def write_min_price_results(f, record_count, cursor, log):
        min_prices_db_count = MinPrice.objects.all().count()

        if options.skip_check:
            log.info('Skip database check')

        elif not test_data(min_prices_db_count, record_count, log):
            log.error('Skip database update for min prices')

            return False

        f.flush()
        os.fsync(f.fileno())

        log.info('Delete from www_minprice_airlines')
        cursor.execute('DELETE FROM www_minprice_airlines')
        auto_inrement_val = check_auto_increment('www_minprice', cursor)

        if auto_inrement_val + record_count < MAX_MYSQL_PK_VALUE:
            log.info('Delete (with exclude) from www_minprice')
            cursor.execute('DELETE FROM www_minprice WHERE eventtime != \'%s\'' % FAKE_EVENTTIME.strftime('%Y-%m-%d'))
        else:
            log.info('Current last id: %s. Set AUTO_INCREMENT to 1 for www_minprice', auto_inrement_val)
            cursor.execute('DELETE FROM www_minprice')
            cursor.execute("ALTER TABLE www_minprice AUTO_INCREMENT = 1")

        log.info('Mysql load data to www_minprice')
        connection.cursor().execute("""
        LOAD DATA LOCAL INFILE %s REPLACE INTO TABLE www_minprice
            CHARACTER SET utf8 FIELDS TERMINATED BY '|'
        (
            departure_settlement_id, arrival_settlement_id, price, currency_id, eventtime, date_forward, national_version, direct_flight, day_of_week, @date_backward, passengers, routes
        )
        SET
        date_backward = NULLIF(@date_backward, 'None')
        """, [f.name])

        min_prices_db_count = MinPrice.objects.all().count()

        log.info('total min prices: %s', min_prices_db_count)

    def aggregate_country_prices():
        log.info('Get aggregated countries prices')
        aggregated_min_prices = MinPrice.objects.all().extra(
            select={
                'date_backward': 'IFNULL(date_backward, \'None\')'
            }
        ).values(
            'departure_settlement__country',
            'arrival_settlement__country',
            'national_version',
            'currency',
            'date_forward',
            'date_backward',
            'passengers',
            'direct_flight',
            'day_of_week'
        ).annotate(
            min_price=Min('price'),
        ).order_by(
            'departure_settlement__country',
            'arrival_settlement__country',
            'national_version',
            'currency',
            'date_forward',
            'date_backward',
            'passengers',
            'direct_flight',
            'day_of_week'
        )

        with tempfile.NamedTemporaryFile() as f:
            log.info('Write country prices to tmp file')
            for x, p in enumerate(aggregated_min_prices):
                line = '%s|%s|%s|%s|%s|%s|%s|%s|%s|%s|%s\n' % (
                    p['departure_settlement__country'],
                    p['arrival_settlement__country'],
                    p['national_version'],
                    p['min_price'],
                    p['currency'],
                    datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
                    p['date_forward'],
                    p['date_backward'],
                    p['passengers'],
                    1 if p['direct_flight'] else 0,
                    p['day_of_week']
                )

                f.writelines(line)

            f.flush()
            os.fsync(f.fileno())

            with transaction.atomic():
                log.info('Delete from www_mincountryprice')
                MinCountryPrice.objects.all().delete()

                log.info('Mysql load data to www_min_country_price')
                connection.cursor().execute("""
                LOAD DATA LOCAL INFILE %s INTO TABLE www_mincountryprice
                    CHARACTER SET utf8 FIELDS TERMINATED BY '|'
                (
                    departure_country_id, arrival_country_id, national_version, price, currency_id, eventtime, date_forward, date_backward, passengers, direct_flight, day_of_week
                )
                SET
                date_backward = NULLIF(@date_backward, 'None')
                """, [f.name])

    def fill_minprices_airlines(cursor):
        def dictfetchall(query):
            cursor.execute(query)
            columns = [col[0] for col in cursor.description]
            return [dict(zip(columns, row)) for row in cursor.fetchall()]

        log.info('Get direct one-way prices')
        min_prices = dictfetchall("SELECT id, routes FROM www_minprice WHERE routes != '';")

        log.info('Parse flight numbers')
        flight_numbers = set([])
        for min_price in min_prices:
            flights = parse_routes(min_price['routes'])
            flight_numbers.update(flights)
        log.info('Parsed %d flight numbers', len(flight_numbers))

        iata_corrector = IATACorrector(logger=log)
        log.info('Get company ids')
        company_ids_by_flight_number = iata_corrector.flight_numbers_to_carriers(flight_numbers)

        log.info('Matching minprices with airlines')
        minprice_airlines = set()
        for min_price in min_prices:
            flights = parse_routes(min_price['routes'])
            for flight in flights:
                airline = company_ids_by_flight_number.get(flight)
                if airline:
                    minprice_airlines.add((min_price['id'], airline))

        with tempfile.NamedTemporaryFile() as f:
            log.info('Write min_prices – airlines to tmp file')

            for minprice_id, airline_id in minprice_airlines:
                f.writelines('{}|{}\n'.format(minprice_id, airline_id))

            f.flush()
            os.fsync(f.fileno())

            auto_inrement_val = check_auto_increment('www_minprice_airlines', cursor)

            if auto_inrement_val + len(minprice_airlines) >= MAX_MYSQL_PK_VALUE:
                log.info('Current last id: %s. Set AUTO_INCREMENT to 1 for www_minprice_airlines', auto_inrement_val)
                cursor.execute("ALTER TABLE www_minprice_airlines AUTO_INCREMENT = 1")

            log.info('Mysql load data to www_minprice_airlines')
            cursor.execute("""
            LOAD DATA LOCAL INFILE %s INTO TABLE www_minprice_airlines
                CHARACTER SET utf8 FIELDS TERMINATED BY '|'
            (
                minprice_id, company_id
            )
            """, [f.name])
            log.info('Data loaded')

    def store_min_prices_with_airlines(tmp_table_min, log):
        cursor = connection.cursor()
        cursor.execute('BEGIN;')

        store_min_prices(tmp_table_min, cursor, log)
        fill_minprices_airlines(cursor)

        cursor.execute('COMMIT;')

    # BEGIN main()
    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", type="days", help="number of last logs to aggregate", default=7)
    optparser.add_option("-n", "--number", dest="quantity", type="quantity", help="quantity of max directions", default=6)
    optparser.add_option("-s", "--skip_check", action="store_true")

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)
        if options.skip_check:
            log.info('Skip database check')

        tmp_table_min = yt.create_temp_table()
        build_min_prices(yt, log, options, tmp_table_min)
        store_min_prices_with_airlines(tmp_table_min, log)

        run_fill_prices(DEFAULT_NATIONAL_COUNTRIES)
        store_avg_prices(tmp_table_min, log)
        aggregate_country_prices()

        yt.remove(tmp_table_min)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')


def build_min_prices(yt, logger, options, destination_table_path):
    from travel.avia.admin.lib.yt_helpers import yt_last_logs_tables

    from travel.avia.library.python.common.models.currency import Currency
    from travel.avia.library.python.common.models.geo import Station
    from travel.avia.library.python.common.models.partner import DohopVendor, Partner

    def process_min_price_record(stations_map, plane_settlements, partners_codes, rates, record):
        today = datetime.now().date()
        left_date_border = today - timedelta(days=options.days)
        date_forward = datetime.strptime(record['date_forward'], '%Y-%m-%d').date()
        eventtime = datetime.fromtimestamp(int(record.get('real_unixtime') or record['unixtime']))

        # Костыль ticket/tickets пока не решили
        service = 'ticket_family' if record['service'] in ['ticket', 'm_avia', 'api_avia'] else 'other'

        good_record = date_forward >= today - timedelta(days=1)

        from_key, to_key = record['direction'].split('_')

        from_key = fix_point(from_key, stations_map)
        to_key = fix_point(to_key, stations_map)

        direct_flight = int(record['stops'] in ['0', '0_0', '0_-1'])

        with_routes = 'forward_routes' in record and 'backward_routes' in record
        economy_klass = 'klass' not in record or record['klass'] == 'economy'
        plane_points = from_key in plane_settlements and to_key in plane_settlements

        filtered_partners = [c for c in record['partners'].split(',') if c in partners_codes]

        positive_date_forward = datetime.strptime(record['date_forward'], '%Y-%m-%d').date() >= left_date_border

        price = float(record.get('original_price') or record['price'])
        currency = record.get('original_currency') or record['currency']

        if price <= 0:
            raise ValueError('Negative price value')

        if all([plane_points, from_key, to_key, positive_date_forward, good_record,
                with_routes, economy_klass, filtered_partners, currency in currency_map]):

            if 'national_version' in record:
                national_version = record['national_version']

            else:
                national_version = record['qid'][-2:]

            direction = '%s_%s' % (from_key, to_key)

            if record['backward_routes']:
                routes = '%s/%s' % (record['forward_routes'], record['backward_routes'])

            else:
                routes = record['forward_routes']

            national_currency = NATIONAL_CURRENCIES.get(national_version)

            if currency != national_currency:
                national_price = (price * FOREIGN_CURRENCY_COEFF) * rates[national_version][currency] / rates[national_version][national_currency]

            else:
                national_price = price

            return {
                'direction': '%s_%s_%s' % (direction, national_version, direct_flight),
                'date_forward': record['date_forward'],
                'date_backward': record['date_backward'] if 'date_backward' in record else '',
                'passengers': record['passengers'] if 'passengers' in record else '1_0_0',
                'direct_flight': direct_flight,
                'routes': routes,
                'iso_eventtime': eventtime.strftime('%Y-%m-%d %H:%M:%S'),
                'price': national_price,
                'currency': national_currency,
                'service': service,
                '@table_index': DESTINATION_TABLE_INDEX,
            }

    def process_unparsed_min_price_record(e):
        return {
            '@table_index': UNPARSED_TABLE_INDEX,
            'error_messsage': e.message
        }

    def unparsed_min_price_reduce(key, records):
        count = 0

        for r in records:
            count += 1

        yield {
            'error_messsage': key['error_messsage'],
            'count': count,
        }

    def process_unparsed_table(tmp_table_unparsed, log):
        unparsed_row_count = int(
            yt.get_attribute(tmp_table_unparsed, 'row_count'))

        if unparsed_row_count:
            log.warning('%s source lines was skipped:' % unparsed_row_count)
            yt.run_sort(
                source_table=tmp_table_unparsed,
                sort_by='error_messsage',
            )

            yt.run_reduce(
                unparsed_min_price_reduce,
                tmp_table_unparsed,
                tmp_table_unparsed,
                reduce_by='error_messsage',
            )

            for record in yt.read_table(tmp_table_unparsed, format=JSON_FORMAT_ENCODE_UTF8_TRUE, raw=False):
                log.warning('%s: %s' % (record['error_messsage'], record['count']))

    def prepare_data(stations_map, plane_settlements, partners_codes,
                     intermediate_table, tmp_table_unparsed, rates, options, log):
        source_tables = yt_last_logs_tables(
            '//home/rasp/logs/rasp-min-price-log', options.days)

        log.info('Prepare prices: %s -> %s', source_tables, intermediate_table)

        yt.run_map(
            functools.partial(min_price_map, stations_map, plane_settlements,
                              partners_codes, rates),
            source_table=source_tables,
            destination_table=[intermediate_table, tmp_table_unparsed],
            format=yt.JsonFormat(attributes={'encode_utf8': True},
                                 control_attributes_mode='row_fields'),
        )

        try:
            process_unparsed_table(tmp_table_unparsed, log)

        except Exception:
            log.exception('ERROR')

        log.info('Sort intermediate table: %s ', intermediate_table)
        yt.run_sort(
            source_table=intermediate_table,
            sort_by=['direction'],
        )

    def build_min_price(intermediate_table, tmp_table, log):
        log.info('Reduce min prices: %s ', tmp_table)
        yt.run_reduce(
            min_price_reduce,
            intermediate_table,
            tmp_table,
            format=JSON_FORMAT_ENCODE_UTF8_TRUE,
            reduce_by='direction',
        )

    def min_price_map(stations_map, plane_settlements, partners_codes, rates, record):
        try:
            min_price_map_record = process_min_price_record(
                stations_map, plane_settlements, partners_codes, rates, record
            )

        except Exception as e:
            min_price_map_record = process_unparsed_min_price_record(e)

        if min_price_map_record:
            yield min_price_map_record

    def min_price_reduce(key, records):
        min_prices = {}
        for r in records:
            date_forward = datetime.strptime(r['date_forward'], '%Y-%m-%d').date()

            if r['date_backward'] and r['date_backward'] != 'None':
                date_backward = datetime.strptime(r['date_backward'], '%Y-%m-%d').date()
            else:
                date_backward = ''

            trip_dates = (date_forward, date_backward)
            iso_eventtime = datetime.strptime(r['iso_eventtime'], '%Y-%m-%d %H:%M:%S')

            price = (
                (datetime.now() - iso_eventtime).total_seconds(),
                float(r['price']),
                r,
                iso_eventtime,
            )

            if trip_dates not in min_prices:
                min_prices[trip_dates] = price

            else:
                stored_record = min_prices[trip_dates][2]
                stored_record_eventtime = min_prices[trip_dates][3]
                if r['service'] == 'ticket_family' or stored_record['service'] != 'ticket_family':
                    if price < min_prices[trip_dates]:
                        min_prices[trip_dates] = price

                elif (iso_eventtime - stored_record_eventtime).total_seconds() > 60 * 60 * 24:
                    min_prices[trip_dates] = price

        for trip_dates, min_price in min_prices.iteritems():
            record = min_price[2]
            date_forward, date_backward = trip_dates

            yield {
                'direction': key['direction'],
                'price': '%s %s' % (record['price'], record['currency']),
                'date_forward': date_forward.strftime('%Y-%m-%d'),
                'date_backward': '' if not date_backward else date_backward.strftime('%Y-%m-%d'),
                'iso_eventtime': record['iso_eventtime'],
                'passengers': record['passengers'],
                'direct_flight': record['direct_flight'],
                'routes': record['routes'],
                'service': record['service']
            }

    def partners_codes_by_type(t_type_code):
        partners = Partner.objects.filter(t_type__code=t_type_code)
        vendors = DohopVendor.objects.filter(enabled=True)

        return [p.code for p in list(chain(partners, vendors))]

    def get_rates():
        currencies_rates = {}
        currencies = Currency.objects.all()

        for national_version, currency_code in NATIONAL_CURRENCIES.items():
            geoid = CAPITALS_MAP.get(national_version)
            _src, rates = Currency.fetch_rates(currencies, geoid)

            currencies_rates[national_version] = rates

        return currencies_rates

    def precached_objects():
        stations = Station.objects.filter(
            t_type__code='plane', settlement_id__isnull=False, hidden=False
        )

        stations = [s for s in stations if s.iata or s.sirena_id]
        station_map = {s.point_key: s.settlement.point_key for s in stations}
        plane_settlements = {s.settlement.point_key for s in stations}
        return station_map, plane_settlements

    stations_map, plane_settlements = precached_objects()
    currency_map = get_currency_map()
    partners_codes = partners_codes_by_type('plane')

    rates = get_rates()

    logger.info('Number of days: %s' % options.days)
    logger.info('%s airports with settlemet loaded' % len(stations_map))
    logger.info('%s plane settlemets loaded' % len(plane_settlements))
    logger.info('%s partners loaded' % len(partners_codes))
    for currency, rate in rates.items():
        logger.info('Rates %s: %s', currency, ', '.join(['%s: %s' % r for r in rate.items()]))

    intermediate_table = yt.create_temp_table()
    tmp_table_unparsed = yt.create_temp_table()

    prepare_data(
        stations_map, plane_settlements, partners_codes,
        intermediate_table, tmp_table_unparsed, rates, options, logger
    )
    build_min_price(intermediate_table, destination_table_path, logger)
    yt.remove(tmp_table_unparsed)
    yt.remove(intermediate_table)


class Yoption(Option):
    TYPES = Option.TYPES + ('days', 'quantity', 'countries')
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER['days'] = check_number
    TYPE_CHECKER['quantity'] = check_number
