# -*- encoding: utf-8 -*-
import functools
import sys

from copy import copy
from optparse import Option, OptionParser, OptionValueError

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger


def point_convert(type, id):
    if type == 'Settlement':
        point_id = 'c%s' % id

    elif type == 'Station':
        point_id = 's%s' % id

    else:
        point_id = None

    return point_id


def parse_price(price):
    try:
        splitted_price = price.split(' ')

    except Exception:
        return None, None

    return splitted_price


def check_number(option, opt, value):
    try:
        days=int(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad number: %r" % (opt, value))

    return days


def test_data(existing_records_count, new_records_count, log=None):
    delta = abs(existing_records_count - new_records_count)

    try:
        percent = int((float(delta) / float(existing_records_count)) * 100)

    except ZeroDivisionError:
        percent = None

    if percent >= 30:
        if log:
            log.error('New/Existing data diff >= 30%%: %s / %s (delta %s) = %s%%' % (
                new_records_count, existing_records_count, delta, percent
            ))

        return False

    elif percent >= 10 and log:
        log.warning('New/Existing data diff >= 10%%: %s / %s (delta %s) = %s%%' % (
            new_records_count, existing_records_count, delta, percent
        ))

    elif log:
        log.info('New/Existing data diff: %s / %s (delta %s) = %s%%' % (
            new_records_count, existing_records_count, delta, percent
        ))

    return True


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.db import transaction
    from django.db.utils import IntegrityError

    from travel.avia.library.python.avia_data.models import TopDirection
    from travel.avia.library.python.common.models.geo import Settlement, Station
    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def reduce_directions(key, records):
        """
        Reduce popular directions
        """
        yield {
            'direction': key['direction'],
            'count': sum([int(r['count']) for r in records])
        }

    def map_directions(stations_map, plane_settlements, record):
        """
        Map popular directions
        """
        def fix_settlement(point):
            if point.startswith('s'):
                try:
                    point = 'c%s' % stations_map[int(point[1:])]

                except Exception:
                    return False

            return point

        try:
            from_id = fix_settlement(record['fromId'])
            to_id = fix_settlement(record['toId'])
            national_version = record['national_version']

        except Exception:
            return

        if to_id in plane_settlements and from_id in plane_settlements:
            if from_id and to_id:
                yield {
                    'direction': '%s_%s_%s' % (from_id, to_id, national_version),
                    'count': 1
                }

    def precached_objects():
        plane_settlements = set()
        stations_map = {}

        stations = Station.objects.filter(
            t_type__code='plane',
            settlement_id__gte=0,
            hidden=False
        )

        for station in stations:
            if station.id not in stations_map:
                stations_map[int(station.id)] = int(station.settlement_id)

        settlements = Settlement.objects.filter(
            type_choices__contains='plane',
            hidden=False,
        )

        for settlement in settlements:
            plane_settlements.add('c%s' % settlement.id)

        return stations_map, plane_settlements

    def build_popular_direction(source_tables, destination_table):
        """
        Run map/reduce to create popular directions table in YT
        """
        log.info('Run popular directions map/reduce')
        source_tables = [
            yt.TablePath(table, columns=['fromId', 'toId', 'national_version'])
            for table in source_tables
        ]
        yt.run_map_reduce(
            mapper=functools.partial(map_directions, stations_map, plane_settlements),
            reducer=reduce_directions,
            source_table=source_tables,
            destination_table=destination_table,
            reduce_by='direction',
        )

    @transaction.atomic
    def write_top_dir_results(source_table, options):
        """
        Read YT and update database
        """
        log.info('Read popular directions from YT')

        top_directions = {}

        for record in yt.read_table(source_table, format=yt.DsvFormat(), raw=False):
            direction = record['direction']
            count = int(record['count'])

            from_id, to_id, national_version = direction.split('_')

            from_key = '%s_%s' % (from_id, national_version)

            if from_key not in top_directions:
                top_directions[from_key] = []

            top_directions[from_key].append((count, to_id))

        log.info('Sort results')
        top_directions_count = 0

        for from_key in top_directions:
            top_directions[from_key] = sorted(top_directions[from_key], reverse=True)[:options.quantity]
            top_directions_count += len(top_directions[from_key])

        if options.skip_check:
            log.info('Skip data check')

        else:
            log.info('Check existing database table with new data')

            test_data_passed = test_data(
                TopDirection.objects.all().count(),
                top_directions_count,
                log=log
            )

            if not test_data_passed:
                log.error('Skip database update for top directions')
                return False

        log.info('Write popular directions to the database')
        TopDirection.objects.all().delete()

        for from_key in top_directions:
            from_id, national_version = from_key.split('_')

            for item in top_directions[from_key]:
                count, to_id = item

                new_top_direction = TopDirection(
                    national_version=national_version,
                    departure_settlement_id=from_id[1:],
                    arrival_settlement_id=to_id[1:],
                    count=count,
                )

                try:
                    new_top_direction.save()

                except IntegrityError:
                    log.warning('Unknown settlemet in pair %s-%s' % (from_id, to_id))

    # BEGIN main()
    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", type="days", help="number of days to aggregate", default=5)
    optparser.add_option("-n", "--number", dest="quantity", type="quantity", help="quantity of ", default=6)
    optparser.add_option("-s", "--skip_check", action="store_true")

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)

        stations_map, plane_settlements = precached_objects()

        log.info('Number of days: %s' % options.days)
        log.info('%s airports with settlemet loaded' % len(stations_map))
        log.info('%s settlemets with airport loaded' % len(plane_settlements))

        if options.skip_check:
            log.info('Skip database check')

        log.info('---------------------------------')

        source_tables = yt_last_logs_tables(
            '//home/avia/logs/avia-users-search-log',
            options.days
        )
        temp_table = yt.create_temp_table()

        build_popular_direction(source_tables, temp_table)
        write_top_dir_results(temp_table, options)

        yt.remove(temp_table)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')


class Yoption(Option):
    TYPES = Option.TYPES + ("days", "quantity")
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER["days"] = check_number
    TYPE_CHECKER["quantity"] = check_number
