# -*- encoding: utf-8 -*-
import functools
import sys
from collections import defaultdict, namedtuple

from copy import copy
from datetime import datetime
from optparse import Option, OptionParser, OptionValueError

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

from travel.avia.library.python.shared_flights_client.client import SharedFlightsClient

TopFlightItem = namedtuple('TopFlightItem', ['count', 'to_id', 'weekday', 'flights'])


def point_convert(type, id):
    if type == 'Settlement':
        point_id = 'c%s' % id

    elif type == 'Station':
        point_id = 's%s' % id

    else:
        point_id = None

    return point_id


def parse_price(price):
    try:
        splitted_price = price.split(' ')

    except Exception:
        return None, None

    return splitted_price


def check_number(option, opt, value):
    try:
        days=int(value)

    except ValueError:
        raise OptionValueError(
            "option %s: bad number: %r" % (opt, value))

    return days


def test_data(existing_records_count, new_records_count, log=None):
    delta = abs(existing_records_count - new_records_count)

    try:
        percent = int((float(delta) / float(existing_records_count)) * 100)

    except ZeroDivisionError:
        percent = None

    if percent >= 30:
        if log:
            log.error('New/Existing data diff >= 30%%: %s / %s (delta %s) = %s%%' % (
                new_records_count, existing_records_count, delta, percent
            ))

        return False

    elif percent >= 10 and log:
        log.warning('New/Existing data diff >= 10%%: %s / %s (delta %s) = %s%%' % (
            new_records_count, existing_records_count, delta, percent
        ))

    elif log:
        log.info('New/Existing data diff: %s / %s (delta %s) = %s%%' % (
            new_records_count, existing_records_count, delta, percent
        ))

    return True


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.db import transaction

    from travel.avia.library.python.avia_data.models import TopFlight
    from travel.avia.library.python.common.models.geo import Settlement, Station
    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper, yt_last_logs_tables

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def reduce_directions(key, records):
        """
        Reduce flights
        """
        top_flights = {}
        for record in records:
            weekday = record['weekday']
            flights = record['flights']
            if weekday not in top_flights:
                top_flights[weekday] = {}

            if flights not in top_flights[weekday]:
                top_flights[weekday][flights] = 0

            top_flights[weekday][flights] += 1

        for weekday in top_flights.keys():
            for flights in top_flights[weekday]:
                yield {
                    'direction': key['direction'],
                    'weekday': weekday,
                    'flights': flights,
                    'count': top_flights[weekday][flights]
                }

    def map_directions(stations_map, plane_settlements, record):
        """
        Map popular directions
        """
        try:
            transport_type_plane = ('transport_type' not in record or record['transport_type'] == 'plane')
            from_point_key = record['from_id']
            to_point_key = record['to_id']
            national_version = record['national_version']
        except Exception:
            return

        if not record['date_forward']:
            return

        date_forward = datetime.strptime(record['date_forward'], '%Y-%m-%d')
        date_backward = datetime.strptime(record['date_backward'], '%Y-%m-%d') if record['date_backward'] else None

        weekday_forward = date_forward.weekday()
        weekday_backward = date_backward.weekday() if date_backward else None

        if transport_type_plane and to_point_key in plane_settlements and from_point_key in plane_settlements:
            yield {
                'direction': '%s_%s_%s' % (from_point_key, to_point_key, national_version),
                'weekday': weekday_forward,
                'flights': record['forward_numbers'],
                'count': 1
            }
            if date_backward:
                yield {
                    'direction': '%s_%s_%s' % (to_point_key, from_point_key, national_version),
                    'weekday': weekday_backward,
                    'flights': record['backward_numbers'],
                    'count': 1
                }

            # Если один из пунктов станция, то обновим статистику для городов
            from_settlement_id = None
            to_settlement_id = None

            if from_point_key.startswith('s'):
                from_settlement_id = stations_map.get(from_point_key[1:])

            if to_point_key.startswith('s'):
                to_settlement_id = stations_map.get(to_point_key[1:])

            if from_settlement_id or to_settlement_id:
                yield {
                    'direction': '%s_%s_%s' % (
                        'c%s' % from_settlement_id if from_settlement_id else from_point_key,
                        'c%s' % to_settlement_id if to_settlement_id else to_point_key,
                        national_version
                    ),
                    'weekday': weekday_forward,
                    'flights': record['forward_numbers'],
                    'count': 1
                }

                if date_backward:
                    yield {
                        'direction': '%s_%s_%s' % (
                            'c%s' % to_settlement_id if to_settlement_id else to_point_key,
                            'c%s' % from_settlement_id if from_settlement_id else from_point_key,
                            national_version
                        ),
                        'weekday': weekday_backward,
                        'flights': record['backward_numbers'],
                        'count': 1
                    }

    def precached_objects():
        plane_settlements = set()
        stations_map = {}

        stations = Station.objects.filter(
            t_type__code='plane',
            settlement_id__gte=0,
            hidden=False
        )

        for station in stations:
            if station.id not in stations_map:
                stations_map[int(station.id)] = int(station.settlement_id)

        settlements = Settlement.objects.filter(
            type_choices__contains='plane',
            hidden=False,
        )

        for settlement in settlements:
            plane_settlements.add('c%s' % settlement.id)

        return stations_map, plane_settlements

    def build_popular_flights(source_tables, destination_table):
        """
        Run map/reduce to create popular directions table in YT
        """
        log.info('Run popular directions map/reduce')
        yt.run_map_reduce(
            mapper=functools.partial(map_directions, stations_map, plane_settlements),
            reducer=reduce_directions,
            source_table=source_tables,
            destination_table=destination_table,
            sort_by=['direction'],
            reduce_by=['direction'],
        )

    def write_top_flights_results(source_table, options):
        """
        Read YT and update database
        """
        log.info('Read popular directions from YT')

        top_flights = {}

        all_flights = defaultdict(set)

        for record in yt.read_table(source_table, format=yt.JsonFormat()):
            direction = record['direction']
            weekday = int(record['weekday'])
            flights = record['flights'].split(';')
            count = int(record['count'])

            from_id, to_id, national_version = direction.split('_')

            from_key = '%s_%s' % (from_id, national_version)

            if from_key not in top_flights:
                top_flights[from_key] = []

            top_flights[from_key].append(TopFlightItem(count, to_id, weekday, flights))

            all_flights[national_version].update(flights)

        shared_flights_client = SharedFlightsClient()
        checked_all_flights = {}
        for national_version, flights in all_flights.iteritems():
            log.info('Check %d flight numbers', len(flights))
            checked_all_flights[national_version] = shared_flights_client.flight_range_accessible(
                flights,
                national_version,
            )

        log.info('Filter unknown flights')
        filtered_top_flights = {}
        for from_key, items in top_flights.iteritems():
            from_id, national_version = from_key.split('_')
            filtered_items = []

            for item in items:
                valid = True
                for flight in item.flights:
                    if not checked_all_flights[national_version].get(flight.decode('utf-8')):
                        log.info('Unknown flight "%s" for "%s"', flight.decode('utf-8'), national_version)
                        valid = False
                if valid:
                    filtered_items.append(item)

            if items:
                filtered_top_flights[from_key] = filtered_items

        top_flights = filtered_top_flights

        log.info('Sort results')
        top_flights_count = 0

        for from_key in top_flights:
            top_flights[from_key] = sorted(top_flights[from_key], reverse=True)
            top_flights_count += len(top_flights[from_key])

        if options.skip_check:
            log.info('Skip data check')

        else:
            log.info('Check existing database table with new data')

            test_data_passed = test_data(
                TopFlight.objects.all().count(),
                top_flights_count,
                log=log
            )

            if not test_data_passed:
                log.error('Skip database update for top directions')
                return False

        top_directions = []
        for from_key in top_flights:
            from_id, national_version = from_key.split('_')

            for item in top_flights[from_key]:
                top_directions.append(TopFlight(
                    from_point_key=from_id,
                    to_point_key=item.to_id,
                    day_of_week=item.weekday,
                    national_version=national_version,
                    redirects=item.count,
                    flights=';'.join(item.flights),
                ))

        with transaction.atomic():
            log.info('Write popular directions to the database')
            TopFlight.objects.all().delete()

            TopFlight.objects.bulk_create(top_directions, batch_size=10000)

    # BEGIN main()
    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", type="days", help="number of logs to aggregate", default=30)
    optparser.add_option("-n", "--number", dest="quantity", type="quantity", help="quantity of ", default=10)
    optparser.add_option("-s", "--skip_check", action="store_true")

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)

        stations_map, plane_settlements = precached_objects()

        log.info('Number of days: %s' % options.days)
        log.info('%s airports with settlemet loaded' % len(stations_map))
        log.info('%s settlemets with airport loaded' % len(plane_settlements))

        if options.skip_check:
            log.info('Skip database check')

        log.info('---------------------------------')

        source_tables = yt_last_logs_tables(
            '//home/rasp/logs/rasp-popular-flights-log',
            options.days
        )

        temp_table = yt.create_temp_table()

        build_popular_flights(source_tables, temp_table)
        write_top_flights_results(temp_table, options)

        yt.remove(temp_table)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')


class Yoption(Option):
    TYPES = Option.TYPES + ("days", "quantity")
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER["days"] = check_number
    TYPE_CHECKER["quantity"] = check_number
