# -*- encoding: utf-8 -*-
import sys
from datetime import datetime, timedelta
from optparse import OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

DESTINATION_WIZARD_MIN_PRICE_TABLE_PATH = '//home/avia/data/wizard/min-price-365d'
ALLOWED_ENVS = ['production', 'dev']
FIRST_DAY_DELTA = 2
LAST_DAY_DELTA = 92


def _parse_date(dt):
    return datetime.strptime(dt, '%Y-%m-%d').date()


def map_table(record):
    direction_parts = record['direction'].split('_')
    if (
        not direction_parts[0].startswith('c')
        and not direction_parts[0].startswith('c')
    ):
        return

    date_forward = _parse_date(record['date_forward'])
    date_backward = _parse_date(record['date_backward']) if record['date_backward'] else None
    today = datetime.today().date()
    if (
        today + timedelta(days=FIRST_DAY_DELTA) <= date_forward <= today + timedelta(days=LAST_DAY_DELTA)
        and (
            not date_backward
            or date_forward + timedelta(days=5) <= date_backward <= date_forward + timedelta(days=14)
        )
    ):
        price, currency_code = record['price'].split(' ')
        yield {
            'price': price,
            'currency__code': currency_code,
            'departure_settlement_id': int(direction_parts[0][1:]),
            'arrival_settlement_id': int(direction_parts[1][1:]),
            'national_version': direction_parts[2],
            'date_forward': record['date_forward'],
            'passengers': record['passengers'],
        }


def reduce_table(key, records):
    min_record = None

    for r in records:
        if not min_record or float(r['price']) < float(min_record['price']):
            min_record = r
    yield min_record


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.conf import settings

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper
    from travel.avia.admin.avia_scripts.build_min_price import build_min_prices

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-d', '--days', dest='days', type='int', help='number of last logs to aggregate', default=7)

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)
    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')
    if settings.ENVIRONMENT not in ALLOWED_ENVS:
        log.info('Environment {} is not allowed', settings.ENVIRONMENT)
        return

    try:
        configure_wrapper(yt)
        if not yt.exists(DESTINATION_WIZARD_MIN_PRICE_TABLE_PATH):
            yt.create('table', path=DESTINATION_WIZARD_MIN_PRICE_TABLE_PATH, recursive=True)

        tmp_table = yt.create_temp_table()
        build_min_prices(yt, log, options, tmp_table)
        yt.run_map_reduce(
            mapper=map_table,
            reducer=reduce_table,
            source_table=tmp_table,
            destination_table=DESTINATION_WIZARD_MIN_PRICE_TABLE_PATH,
            reduce_by=[
                'departure_settlement_id',
                'arrival_settlement_id',
                'national_version',
                'currency__code',
                'date_forward',
            ],
        )
        yt.remove(tmp_table)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')
