# -*- encoding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import getpass
import logging
import sys
from collections import OrderedDict
from copy import copy
from itertools import chain
from operator import attrgetter
from optparse import OptionParser

import six
import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger
from django.conf import settings
from django.db.models import Q
from six.moves import reload_module

from travel.avia.library.python.avia_data.models import AdminSettlementBigImage, AviaCompany, AmadeusMerchant
from travel.avia.library.python.common.models.geo import Settlement, Station, StationCode, Country, CodeSystem, Station2Settlement
from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType

from travel.avia.admin.lib.feature_flags import new_pricing_flag_by_partner_code, extended_report_flag_by_partner_code
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
from travel.avia.admin.lib.yt_helpers import configure_wrapper

ALLOWED_ENVS = ['production', 'dev']
if settings.ENVIRONMENT != 'dev':
    YT_REFERENCE_PATH = '//home/rasp/reference/'
    TOLOKA_REFERENCE_PATH = '//home/avia/toloka/aviacompany'
else:
    user = getpass.getuser()
    YT_REFERENCE_PATH = '//home/avia/dev/{}/reference/'.format(user)
    TOLOKA_REFERENCE_PATH = '//home/avia/dev/{}/toloka/aviacompany'.format(user)

SIRENA_CODE_SYSTEM_ID = CodeSystem.objects.get(code='sirena').id
IATA_CODE_SYSTEM_ID = CodeSystem.objects.get(code='iata').id
ICAO_CODE_SYSTEM_ID = CodeSystem.objects.get(code='icao').id


def _make_schema_for_toloka(schema, exclude_fields, prefix='reviewed_'):
    target_schema = copy(schema)
    target_schema.extend(
        {
            'type': item['type'],
            'name': prefix + item['name'],
        }
        for item in schema
        if item['name'] not in exclude_fields
    )

    return target_schema


def make_table_for_toloka(src_path, dst_path, exclude_fields):
    with yt.Transaction():
        if yt.exists(dst_path):
            yt.remove(dst_path)

        schema = yt.get_attribute(src_path, 'schema')
        yt.create(
            'table',
            dst_path,
            attributes={
                'optimize_for': 'scan',
                'schema': _make_schema_for_toloka(schema, exclude_fields=exclude_fields)
            }
        )

        yt.run_merge(
            src_path,
            dst_path,
            spec={
                'schema_inference_mode': 'from_output',
            },
        )


class YtModel(object):
    def __init__(self, model, fields, getter=None, types=None, table=''):
        """

        :param django.db.models.Model model: Джанго-модель
        :param fields: Список полей в YT
        :param dict getter: Функции для получения итоговых полей
        :param dict types: типы для полей из fields
        :param basestring table: Название таблицы. По умолчанию совпадает с названием модели
        """
        self.model = model
        self.fields = fields
        self.getter = getter or {}
        self.types = types or {}
        self.yt_path = YT_REFERENCE_PATH + (table or self.model.__name__.lower())
        self.schema = self._get_scheme()

    def __repr__(self):
        return 'YtModel for %r' % self.model

    def save(self, yt, objects):
        with yt.Transaction():
            if yt.exists(self.yt_path):
                yt.remove(self.yt_path)

            yt.create('table', self.yt_path, recursive=True, attributes={
                'optimize_for': 'scan',
                'schema': self.schema,
            })
            yt.write_table(self.yt_path, self._rows(objects))

    def _rows(self, objects):
        for obj in objects:
            yield {
                field: self.getter.get(field, lambda obj: getattr(obj, field))(obj)
                for field in self.fields
            }

    def _get_scheme(self):
        return [
            {
                'type': self.types.get(field, self._model_field_type(field)),
                'name': field
            }
            for field in self.fields
        ]

    @staticmethod
    def _model_field_type(field):
        return 'string'


def tariff_getter_fabric(field):
    def inner(aviacompany):
        # Now we suppose that every airline has no more than one tariff
        tariff = aviacompany.tariffs.first()
        if tariff:
            return getattr(tariff, field)

        return None

    return inner


def bool_to_letter(value):
    return 'Y' if value else 'N'


def _get_station_codes(code_system):
    code_system_id = CodeSystem.objects.get(code=code_system).id
    return dict(StationCode.objects.filter(system_id=code_system_id).values_list('station_id', 'code'))


def build_station_queryset():
    return Station.objects.filter(
        Q(t_type_id=TransportType.PLANE_ID)
        | (
            Q(code_set__system_id__in=(IATA_CODE_SYSTEM_ID, SIRENA_CODE_SYSTEM_ID, ICAO_CODE_SYSTEM_ID))
            & ~Q(code_set__code__in=(None, ''))
        )
    ).select_related('t_type').distinct()


def build_export_models():
    airline_cost_types = dict(AviaCompany.objects.values_list('rasp_company_id', 'cost_type'))
    station_iata_codes = _get_station_codes('iata')
    station_sirena_codes = _get_station_codes('sirena')
    station_icao_codes = _get_station_codes('icao')
    sirena_getter = lambda s: s.sirena_id.encode('utf-8') if s.sirena_id else ''
    title_getter = lambda s: s.L_title().encode('utf-8').strip()
    geo_id_getter = lambda obj: obj._geo_id or 0

    partners_getters = {
        'title': title_getter,
        'billing_client_id': lambda s: s.billing_client_id if isinstance(s, Partner) else None,
        'billing_order_id': lambda s: s.billing_order_id if isinstance(s, Partner) else None,
        'can_fetch_by_daemon': lambda s: s.can_fetch_by_daemon if isinstance(s, Partner) else getattr(s, 'enabled'),
        'disabled': lambda s: s.disabled if isinstance(s, Partner) else None,
        'partner_id': lambda s: s.id if isinstance(s, Partner) else None,
        'use_in_update_conversions': lambda s: s.use_in_update_conversions if isinstance(s, Partner) else None,
        'new_pricing_flag': lambda s: new_pricing_flag_by_partner_code(s.code) if isinstance(s, Partner) else None,
        'extended_report_flag': lambda s: extended_report_flag_by_partner_code(s.code) if isinstance(s, Partner) else None,
    }

    partners_fields = (
        'billing_client_id', 'billing_order_id', 'review_percent', 'can_fetch_by_daemon', 'disabled',
        'code', 'title', 'partner_id',
        'enabled_in_ticket_ru', 'enabled_in_ticket_ua', 'enabled_in_ticket_kz',
        'enabled_in_ticket_com', 'enabled_in_ticket_tr', 'enabled_in_mobile_ticket_ru',
        'enabled_in_mobile_ticket_ua', 'enabled_in_mobile_ticket_kz',
        'enabled_in_mobile_ticket_com', 'enabled_in_mobile_ticket_tr',
        'use_in_update_conversions',
        'new_pricing_flag', 'extended_report_flag',
        'pricing_model',
    )

    partners_field_types = {
        'billing_client_id': 'int64',
        'billing_order_id': 'int64',
        'partner_id': 'int64',
        'review_percent': 'int64',
        'can_fetch_by_daemon': 'boolean',
        'disabled': 'boolean',
        'enabled_in_ticket_ru': 'boolean',
        'enabled_in_ticket_ua': 'boolean',
        'enabled_in_ticket_kz': 'boolean',
        'enabled_in_ticket_com': 'boolean',
        'enabled_in_ticket_tr': 'boolean',
        'enabled_in_mobile_ticket_ru': 'boolean',
        'enabled_in_mobile_ticket_ua': 'boolean',
        'enabled_in_mobile_ticket_kz': 'boolean',
        'enabled_in_mobile_ticket_com': 'boolean',
        'enabled_in_mobile_ticket_tr': 'boolean',
        'use_in_update_conversions': 'boolean',
        'new_pricing_flag': 'boolean',
        'extended_report_flag': 'boolean',
        'pricing_model': 'string',
    }
    yt_avia_company = (
        YtModel(
            AviaCompany,
            (
                'id',
                'iata',
                'cost_type',
                'baggage_rules',
                'baggage_rules_are_valid',
                'baggage_rules_url',
                'baggage_length',
                'baggage_width',
                'baggage_height',
                'baggage_dimensions_sum',
                'carryon_length',
                'carryon_width',
                'carryon_height',
                'carryon_dimensions_sum',

                'baggage_allowed',
                'baggage_norm',
                'carryon',
                'carryon_norm',

                'title',

            ),
            getter={
                'id': attrgetter('pk'),
                'baggage_allowed': lambda x: bool_to_letter(tariff_getter_fabric('baggage_allowed')(x)),
                'baggage_norm': tariff_getter_fabric('baggage_norm'),
                'carryon': lambda x: bool_to_letter(tariff_getter_fabric('carryon')(x)),
                'carryon_norm': tariff_getter_fabric('carryon_norm'),
                'title': lambda x: x.rasp_company.title
            },
            types={
                'id': 'int64',
                'iata': 'string',
                'cost_type': 'string',
                'baggage_rules': 'string',
                'baggage_rules_are_valid': 'boolean',
                'baggage_rules_url': 'string',
                'baggage_length': 'int64',
                'baggage_width': 'int64',
                'baggage_height': 'int64',
                'baggage_dimensions_sum': 'int64',
                'carryon_length': 'int64',
                'carryon_width': 'int64',
                'carryon_height': 'int64',
                'carryon_dimensions_sum': 'int64',
                'baggage_allowed': 'string',
                'baggage_norm': 'double',
                'carryon': 'string',
                'carryon_norm': 'double',
                'title': 'string',
            },
        ),
        AviaCompany.objects.all().select_related('rasp_company').prefetch_related('tariffs'),
    )

    models = OrderedDict(
        (
            (
                YtModel(
                    Station,
                    (
                        'id', 'title', 'iata', 'sirena', 'icao', 'city_id', 'country_id', 'time_zone',
                        'longitude', 'latitude', 't_type', 'hidden',
                    ),
                    getter={
                        'id': lambda s: s.point_key,
                        'title': title_getter,
                        'iata': lambda s: station_iata_codes.get(s.id, ''),
                        'sirena': lambda s: station_sirena_codes.get(s.id, ''),
                        'icao': lambda s: station_icao_codes.get(s.id, ''),
                        'city_id': lambda s: 'c%d' % s.settlement_id if s.settlement_id else None,
                        'country_id': lambda s: 'l%d' % s.country_id if s.country_id else None,
                        't_type': lambda s: s.t_type.code,
                    },
                    types={
                        'longitude': 'double',
                        'latitude': 'double',
                        'hidden': 'boolean',
                    }
                ),
                build_station_queryset(),
            ),
            (
                YtModel(
                    Settlement,
                    (
                        'id', 'title', 'iata', 'sirena', 'country_id', 'majority', 'geo_id', 'longitude', 'latitude',
                        'hidden',
                    ),
                    getter={
                        'id': lambda s: s.point_key,
                        'title': title_getter,
                        'iata': lambda s: s.iata or '',
                        'sirena': sirena_getter,
                        'majority': lambda s: s.majority_id,
                        'country_id': lambda s: 'l%d' % s.country_id if s.country_id else None,
                        'geo_id': geo_id_getter,
                    },
                    types={
                        'majority': 'int64',
                        'geo_id': 'int64',
                        'longitude': 'double',
                        'latitude': 'double',
                        'hidden': 'boolean',
                    },
                ),
                Settlement.objects.all(),
            ),
            (
                YtModel(
                    Country,
                    ('id', 'title', 'code', 'geo_id'),
                    getter={
                        'id': lambda s: s.point_key,
                        'title': title_getter,
                        'geo_id': geo_id_getter,
                    },
                    types={
                        'geo_id': 'int64',
                    },
                ),
                Country.objects.all(),
            ),
            (
                YtModel(
                    Company,
                    (
                        'id',
                        'title',
                        'iata',
                        'sirena',
                        'cost_type',
                        'priority',
                        'url',
                        'registration_url',
                        'registration_url_ru',
                        'registration_url_en',
                        'registration_url_tr',
                        'registration_url_uk',
                        'hidden',
                    ),
                    getter={
                        'title': title_getter,
                        'iata': lambda s: s.iata or '',
                        'sirena': sirena_getter,
                        'cost_type': lambda s: airline_cost_types.get(s.id),
                    },
                    types={
                        'id': 'int64',
                        'priority': 'int64',
                        'registration_url': 'string',
                        'registration_url_ru': 'string',
                        'registration_url_en': 'string',
                        'registration_url_tr': 'string',
                        'registration_url_uk': 'string',
                        'hidden': 'boolean',
                    },
                ),
                Company.objects.filter(t_type_id=TransportType.PLANE_ID),
            ),
            (
                YtModel(
                    Partner,
                    partners_fields,
                    getter=partners_getters,
                    types=partners_field_types,
                ),
                chain(
                    Partner.objects.all(),
                    DohopVendor.objects.all(),
                    AmadeusMerchant.objects.all(),
                ),
            ),
            (
                YtModel(
                    AdminSettlementBigImage,
                    ('settlement', 'url2'),
                    getter={
                        'settlement': lambda s: 'c{}'.format(s.settlement.id) if s.settlement else None,
                        'url2': lambda img: img.url2.url,
                    },
                    types={
                        'settlement': 'string',
                        'url2': 'string',
                    },
                ),
                AdminSettlementBigImage.objects.all(),
            ),
            (
                YtModel(
                    Station2Settlement,
                    ('station_id', 'city_id'),
                    getter={
                        'station_id': lambda s: 's%d' % s.station_id if s.station_id else None,
                        'city_id': lambda s: 'c%d' % s.settlement_id if s.settlement_id else None,
                    },
                ),
                Station2Settlement.objects.all(),
            ),
            yt_avia_company,
        )
    )
    return models


def main():
    log = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-e', '--entities')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout(log)
    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload_module(yt_logger)

    log.info('Start')

    entities_to_update = None if options.entities is None else options.entities.split(',')
    log.info('Will update %s models.', 'all' if entities_to_update is None else str(entities_to_update))

    current_env = settings.ENVIRONMENT
    if current_env not in ALLOWED_ENVS:
        allowed_envs_str = ', '.join(ALLOWED_ENVS)
        log.info('Current ENVIRONMENT %s. Run only %s allowed.', current_env, allowed_envs_str)
        sys.exit()

    configure_wrapper(yt)

    models = build_export_models()

    try:
        for model, objects in six.iteritems(models):
            if entities_to_update is None or model.model.__name__.lower() in entities_to_update:
                log.info('Saving %r', model)
                model.save(yt, objects)

        yt_avia_company_model = next(model for model in six.iterkeys(models) if model.model == AviaCompany)
        if entities_to_update is None or yt_avia_company_model.model.__name__.lower() in entities_to_update:
            log.info('Make AviaCompany for toloka')
            make_table_for_toloka(
                yt_avia_company_model.yt_path,
                TOLOKA_REFERENCE_PATH,
                exclude_fields={'id', 'iata'},
            )
    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')
