# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
from operator import itemgetter

from django.db import transaction

from travel.avia.library.python.avia_data.models import TopFlight, RelevantCityAirline, NationalVersion
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.utils.iterrecipes import group_by
from travel.avia.library.python.iata_correction import IATACorrector

from travel.avia.admin.avia_scripts.utils.argument_parser import verbose_argument_parser
from travel.avia.admin.lib.logs import create_current_file_run_log

log = logging.getLogger(__name__)


def get_relevant_cities():
    national_versions = dict(NationalVersion.objects.all().values_list('code', 'id'))
    settlement_ids = set(Settlement.objects.all().values_list('id', flat=True))

    top_flights_by_nv = group_by(
        TopFlight.objects.all().values('national_version', 'flights', 'to_point_key', 'redirects'),
        itemgetter('national_version')
    )

    iata_corrector = IATACorrector(logger=log)

    for nv, top_flights in top_flights_by_nv:
        if nv not in national_versions:
            log.error('Unknown national_version %s', nv)
            continue
        nv_id = national_versions[nv]

        direct_flights = filter(lambda tf: ';' not in tf['flights'], top_flights)

        company_ids_by_flight_number = iata_corrector.flight_numbers_to_carriers(
            set([f['flights'] for f in direct_flights])
        )

        for point_to_key, flights in group_by(direct_flights, itemgetter('to_point_key')):
            if not point_to_key.startswith('c'):
                continue
            settlement_to_id = int(point_to_key[1:])
            if settlement_to_id not in settlement_ids:
                log.warning('Unknown settlement id %d', settlement_to_id)
                continue

            for airline_id, top_flights in group_by(
                    flights,
                    lambda top_flight: company_ids_by_flight_number.get(top_flight['flights'])
            ):
                if not airline_id:
                    continue
                try:
                    airline = Company.objects.get(id=airline_id)
                except Company.DoesNotExist:
                    log.warning('Company (id=%d) not found', airline_id)
                    continue
                yield RelevantCityAirline(
                    national_version_id=nv_id,
                    settlement_to_id=settlement_to_id,
                    airline_id=airline.id,
                    score=sum(map(itemgetter('redirects'), top_flights))
                )


@transaction.atomic
def update_relevant_city_airlines():
    RelevantCityAirline.objects.all().delete()
    created = RelevantCityAirline.objects.bulk_create(
        get_relevant_cities(), batch_size=1000
    )
    log.info('Stored %d', len(created))


def _main():
    log.info('Start')
    update_relevant_city_airlines()
    log.info('Done')


def main():
    create_current_file_run_log()
    verbose_argument_parser.parse_args()
    _main()
