# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
from collections import defaultdict
from operator import attrgetter
from optparse import OptionParser

from travel.avia.library.python.avia_data.models import FlightRating, CompanyRating
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.iata_correction import IATACorrector

from travel.avia.admin.lib.logs import create_current_file_run_log, print_log_to_stdout

log = logging.getLogger(__name__)


def flights_by_airline():
    airline_flights = defaultdict(list)

    iata_corrector = IATACorrector(logger=log)

    all_flights = FlightRating.objects.all()

    company_ids_by_flight_number = iata_corrector.flight_numbers_to_carriers(
        [flight.number for flight in all_flights]
    )

    for flight in all_flights:
        company_id = company_ids_by_flight_number.get(flight.number)
        if not company_id:
            continue
        try:
            company = Company.objects.get(id=company_id)
        except Company.DoesNotExist:
            log.warning('Company (id=%d) not found', company_id)
            continue

        airline_flights[company].append(flight)

    return dict(airline_flights)


def summarize(attr, iterable):
    return sum(map(attrgetter(attr), iterable))


def update_company_rating(airline_flights):
    log.info('Updating airlines rating')
    min_score, max_score = 10, 0
    airline_ratings = {}
    for company, ratings in airline_flights.iteritems():
        good = summarize('good_count', ratings)
        bad = summarize('bad_count', ratings)
        total = good + bad
        scores = summarize('scores', ratings)
        avg_score = float(scores) / total
        min_score = min(min_score, avg_score)
        max_score = max(max_score, avg_score)
        airline_ratings[company] = dict(
            flight_count=len(ratings),
            scores=scores,
            good_count=good,
            bad_count=bad,
            bad_percent=bad * 100. / total,
            avg_scores=avg_score,
            outrunning=summarize('outrunning', ratings) / len(ratings),
            delayed_less_30=summarize('delayed_less_30', ratings) / len(ratings),
            delayed_30_60=summarize('delayed_30_60', ratings) / len(ratings),
            delayed_60_90=summarize('delayed_60_90', ratings) / len(ratings),
            delayed_more_90=summarize('delayed_more_90', ratings) / len(ratings),
            canceled=summarize('canceled', ratings) / len(ratings),
        )
    interval_length = max_score - min_score
    for company, rating in airline_ratings.iteritems():
        rating['avg_scores'] = min(10, max(0, (rating['avg_scores'] - min_score)) * 10. / interval_length)
        CompanyRating.objects.update_or_create(
            company=company,
            defaults=rating
        )


def _main():
    log.info('Start')
    airline_flights = flights_by_airline()
    update_company_rating(airline_flights)
    log.info('Done')


def main():
    create_current_file_run_log()
    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()
    if options.verbose:
        print_log_to_stdout()
    _main()
