# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import json
import logging
import os
import requests
import socket
import sys
import time

from collections import defaultdict
from datetime import datetime
from optparse import OptionParser
from requests.exceptions import RequestException
from threading import Thread
from time import sleep
from Queue import Queue

from django.conf import settings
from django.core.mail.message import EmailMultiAlternatives

from travel.avia.library.python.common.models.geo import Country, Station, Settlement, CityMajority
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.simple_daemon_client import simple_daemon_client
from travel.avia.admin.stats.models import MinPriceStatus


BACK_API_TEMPLATE = '{avia_back_url}/?national_version=ru&lang=ru&front_host=avia.yandex.ru&name={name}'

RECIPES_URL = BACK_API_TEMPLATE.format(
    avia_back_url=settings.AVIA_BACK_URL,
    name='receipes'
)

MAX_THREADS = 10
WAITING_TIME = 190
MAX_CITIES = 6
WARNING_PERCENT = 75
MAX_RETRY = 5
RECIPIENTS = ['avia-process@yandex-team.ru',
              'bfa@yandex-team.ru']
ALLOWED_ENVS = ['production', 'dev']


q = Queue()
log = logging.getLogger(__name__)
create_current_file_run_log()
results = defaultdict(list)
statuses = defaultdict(int)


def city_title(city_id):
    try:
        s = Settlement.objects.get(id=city_id)

    except Settlement.DoesNotExist:
        return None

    return s.L_title()


def get_backend_json(url, data):
    retry_count = 0
    while True:
        retry_count += 1
        try:
            headers = {
                'user-agent': 'avia-admin',
                'content-type': 'application/json',
            }

            r = requests.post(
                url,
                data=json.dumps(data),
                headers=headers
            )

        except RequestException as e:
            log.warning('Error getting post url: %s: %s' % (url, e.message))
            sleep(5)
            continue

        if retry_count >= MAX_RETRY:
            return None

        if r.status_code != requests.codes.ok:
            log.warning('Bad status code %s for %s' % (r.status_code, url))
            sleep(5)
            continue

        if r.status_code == requests.codes.ok:
            return r.json()


def get_city(url, city_id):
    data = [
        {
            "name": "recipes",
            "params": {
                "when": None,
                "adultSeats": 1,
                "childrenSeats": 0,
                "infantSeats": 0,
                "klass": "economy",
                "fromId": city_id
            },
            "fields": [
                {
                    "name": "recipeOffers",
                    "params": {
                        "recipeId": 0,
                        "extraIds": [],
                        "limit": 36
                    },
                    "fields": [
                        {
                            "name": "fromCity",
                            "fields": [
                                "key",
                            ]
                        },
                        {
                            "name": "toCity",
                            "fields": [
                                "key",
                            ]
                        },
                        "dateForward",
                        "dateBackward",
                        "directPrice",
                        "transfersPrice",
                    ]
                }
            ]
        }
    ]

    return get_backend_json(
        url,
        data
    )


def process_city(q):
    while True:
        try:
            task_number, tasks_total, city_id = q.get()

            # Засыпаем, что бы не шарахнуть по демону разом всеми тредами
            if task_number < MAX_THREADS:
                sleep(task_number * 5)

            prices = []

            city_json = get_city(RECIPES_URL, city_id)
            data_items = city_json['data'][0]

            for data_item in data_items:
                recipe_offers = data_item['recipeOffers']

                if not recipe_offers:
                    continue

                for recipe_offer in recipe_offers:
                    if len(prices) >= MAX_CITIES:
                        break

                    transfers_price = recipe_offer.get('transfersPrice')
                    direct_price = recipe_offer.get('directPrice')

                    min_prices = filter(None, [
                        transfers_price['value'] if transfers_price else None,
                        direct_price['value'] if direct_price else None,
                    ])

                    if not min_prices:
                        continue

                    min_price = min(min_prices)

                    to_point_key = recipe_offer['toCity']['key']

                    prices.append({
                        'city_to': city_title(to_point_key[1:]),
                        'price': min_price,
                        'query_key': '{from_city}_{to_ity}_{when}_{return_date}_economy_1_0_0_ru'.format(
                            from_city=recipe_offer['fromCity']['key'],
                            to_ity=to_point_key,
                            when=recipe_offer['dateForward'],
                            return_date=recipe_offer['dateBackward'],
                        )
                    })

            log.info(u'%s / %s: Проверяем для города %s' % (
                task_number + 1, tasks_total, city_title(city_id))
            )

            for price in prices:
                # Инициализируем поиск
                query_key = price['query_key']
                splitted_query_key = query_key.split('_')
                from_point_key = splitted_query_key[0]
                to_point_key = splitted_query_key[1]
                when = splitted_query_key[2] or None
                return_date = splitted_query_key[3] or None
                params = {
                    "clientCityId": 213,
                    "mobile": False,
                    "clientIpCountryGeoId": 225,
                    "point_from": from_point_key,
                    "point_to": to_point_key,
                    "date_forward": when,
                    "klass": "economy",
                    "adults": 1,
                    "children": 0,
                    "infants": 0,
                    "service": "ticket",
                    "lang": "ru",
                    "national": "ru"
                }
                if return_date and return_date != 'None':
                    params["date_backward"] = return_date

                try:
                    real_min_price = simple_daemon_client.find_min_price(params, 'RUR')

                except ValueError:
                    real_min_price = None

                if real_min_price is None:
                    status = 'Bad answer'

                elif real_min_price > price['price'] * 1.1:
                    status = 'Greater'

                elif real_min_price < price['price'] * 0.9:
                    status = 'Lower'

                else:
                    status = 'In range'

                price['status'] = status
                price['morda_price'] = price['price']
                price['real_price'] = real_min_price
                price['from_point_key'] = from_point_key
                price['to_point_key'] = to_point_key

                results[city_id].append(price)

        except Exception:
            log.exception('WARNING')

        q.task_done()


def get_cities_ids(majority, country_codes):
    stations = Station.objects.filter(
        t_type__code='plane',
        settlement_id__isnull=False,
        hidden=False,
        settlement__majority__lte=majority,
        country__code__in=country_codes,
    )

    return {int(s.settlement.id) for s in stations}


def send_email(receipents, body_lines, percent):
    body = '\n'.join(body_lines)

    if percent < WARNING_PERCENT:
        subject = u'ВНИМАНИЕ! Проверка популярных направлений: %s%%' % percent

    else:
        subject = u'Проверка популярных направлений'

    mail = EmailMultiAlternatives(
        subject=subject,
        body=body,
        from_email=settings.SERVER_EMAIL,
        to=receipents,
    )

    mail.send()


def get_country(country_code):
    try:
        return Country.objects.get(code=country_code)

    except Country.DoesNotExist:
        return None


def save_results(results, country_code):
    for city_id, price in results.items():
        for p in price:
            mps, created = MinPriceStatus.objects.get_or_create(
                eventdate=datetime.now(),
                national_version=country_code,
                settlement_from=Settlement.get_by_key(p['from_point_key']),
                settlement_to=Settlement.get_by_key(p['to_point_key']),
            )

            mps.status = p['status']
            mps.save()


def make_report(results, country_code):
    country = get_country(country_code)
    message_lines = []
    totals_lines = []

    for city_id, price in results.items():
        percents = []
        lines = []
        # Подготовка
        for p in price:
            lines.append(u'      %s - %s: %s %s' % (
                city_title(city_id),
                p['city_to'],
                p['status'].upper(),
                '(%s | %s)' % (p['morda_price'], p['real_price']) if p['status'] in ['Lower', 'Greater'] else '',
            ))

            percents.append(
                1 if p['status'] == 'In range' else 0
            )

            statuses[p['status']] += 1

        percent = int(float(sum(percents)) / len(percents) * 100)

        line = (u'--- [ %s: %s%% ] ---' % (
            city_title(city_id).upper(),
            percent
        )).ljust(60, '-')

        message_lines.append('')
        message_lines.append(line)

        for line in lines:
            message_lines.append(line)

    # Считаем общий процент
    total_statuses = sum(statuses.values())

    message_lines.append(u'')

    for status, count in statuses.items():
        totals_lines.append(u'%s: %s%%' % (status, int(float(count) / total_statuses * 100)))

    totals_lines.append('')

    totals_lines.append(u'Страна: %s' % country.L_title())
    totals_lines.append(u'Городов: %s' % len(results.keys()))
    totals_lines.append(u'Направлений: %s' % total_statuses)
    totals_lines.append('')

    total_in_range_percent = int(float(statuses['In range']) / total_statuses * 100) if total_statuses else 0

    return totals_lines + message_lines, total_in_range_percent


def send_graphite(percent):
    hostname = socket.gethostname()
    if hostname.find('.dev') >= 0:
        log.info('Doest not send in this environemnt')
        return

    t = int(time.time())
    try:
        os.system('echo geo.avia.avia_admin.{}.min_price_check_percent {} {} | nc localhost 42000'.format(
            hostname.replace('.', '_'),
            int(percent),
            t
        ))
    except:
        log.exception('ERROR:')
        raise


def main():
    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('--skip_env_check', action='store_true')
    optparser.add_option('-c', '--country', default='ru')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('*** Start')

    current_env = settings.ENVIRONMENT
    if current_env not in ALLOWED_ENVS and not options.skip_env_check:
        allowed_envs_str = ', '.join(ALLOWED_ENVS)
        log.info('Current ENVIRONMENT %s. Run only %s allowed.' % (current_env, allowed_envs_str))
        sys.exit()

    if not settings.AVIA_BACK_URL:
        log.info('AVIA_BACK_URL not configured. Exit.')
        sys.exit()

    cities = get_cities_ids(
        CityMajority.POPULATION_MILLION_ID,
        [options.country]
    )

    threads_num = len(cities) if len(cities) < MAX_THREADS else MAX_THREADS
    log.info('*** Runing %s threads' % threads_num)

    for i in range(threads_num):
        worker = Thread(target=process_city, args=(q, ))
        worker.setDaemon(True)
        worker.start()

    for x, city_id in enumerate(cities):
        q.put((x, len(cities), city_id))

    log.info(u'*** В базе %s городов-миллионников' % len(cities))
    q.join()

    log.info('*** Make results')
    lines, percent = make_report(results, options.country)

    log.info('*** Save results')
    save_results(results, options.country)

    log.info('*** Email results')
    send_email(RECIPIENTS, lines, percent)

    log.info('*** Send results to graphite')
    send_graphite(percent)

    log.info('*** Done')
