# -*- encoding: utf-8 -*-
import sys

from collections import defaultdict
from datetime import datetime, date, timedelta
from optparse import OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

ALLOWED_ENVS = ['production', 'dev']
RECEPIENTS = {
    'dayly': ['bfa@yandex-team.ru', 'arancio@yandex-team.ru'],
    'hourly': ['avia-dev@yandex-team.ru', 'arancio@yandex-team.ru']
}
HOURLY_MAX_TIMEDELTA = timedelta(hours=1.5)


def path_filter(path):
    return path.split('/')[-2].startswith('rasp')


def get_last_yt_logs(logs_path):
    tables = defaultdict(list)

    for table in yt.search(logs_path, node_type=['table'], path_filter=path_filter, depth_bound=2):
        table_name = table.split('/')[-2]
        try:
            table_date = datetime.strptime(table.split('/')[-1], '%Y-%m-%d').date()
        except ValueError:
            continue

        if table_date == date.today():
            continue

        tables[table_name].append(table)
        tables[table_name] = sorted(tables[table_name], reverse=True)[:2]

    return dict(tables)


def convert_eventtime(record):
    iso_eventtime = record.get('iso_eventtime')

    if not iso_eventtime:
        return

    try:
        iso_eventtime = datetime.strptime(iso_eventtime, '%Y-%m-%d %H:%M:%S')
    except ValueError:
        return

    yield {
        'eventtime': iso_eventtime.strftime('%Y-%m-%d %H:%M')
    }


def aggregate_eventtime(key, records):
    yield {
        "eventtime": key["eventtime"]
    }


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.conf import settings
    from django.core.mail.message import EmailMultiAlternatives

    from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def send_email(recipients, body_lines):
        body = '\n'.join(body_lines)
        subject = u'Проверка YT'

        mail = EmailMultiAlternatives(
            subject=subject,
            body=body,
            from_email=settings.SERVER_EMAIL,
            to=recipients,
        )

        mail.send()

    def check_dayly():
        paths= ['//home/rasp/logs']
        yesterday_str = (date.today() - timedelta(days=1)).strftime('%Y-%m-%d')
        body_lines = []

        for path in paths:
            tables = get_last_yt_logs(path)

            for table, last_tables in tables.items():
                table_path = '%s/%s' % (path, table)

                yesterday_log_date = last_tables[0].split('/')[-1]
                day_before_yesterday_log_date = last_tables[1].split('/')[-1]

                if not last_tables[0].endswith(yesterday_str):
                    line = '%s:\tTABLE NOT EXISTS (last: %s)' % (table_path, yesterday_log_date)
                    body_lines.append(line)
                    log.info(line)
                    continue

                if len(last_tables) != 2:
                    line = '%s:\tONLY ONE LOG FOUND', table_path
                    body_lines.append(line)
                    log.info(line)
                    continue

                day_before_yesterday_rows_count = yt.get_attribute(last_tables[1], 'row_count')
                yesterday_log_rows_count = yt.get_attribute(last_tables[0], 'row_count')

                if not int(day_before_yesterday_rows_count):
                    line = '%s:\tEMPTY TABLE (last: %s)' % (table_path, day_before_yesterday_log_date)
                    body_lines.append(line)
                    log.info(line)
                    continue

                rows_count_delta = int(yesterday_log_rows_count) - int(day_before_yesterday_rows_count)
                rows_count_delta_percent = (float(rows_count_delta) / day_before_yesterday_rows_count) * 100

                rows_count_delta_sigh = "+" if rows_count_delta > 0 else ""
                rows_count_delta_percent_sign = "+" if rows_count_delta_percent > 0 else ""

                line = '%s:\t%s%s (%s%.2f %%) (%s - %s)' % (
                    table_path,
                    rows_count_delta_sigh, rows_count_delta,
                    rows_count_delta_percent_sign, rows_count_delta_percent,
                    day_before_yesterday_log_date, yesterday_log_date
                )
                body_lines.append(line)
                log.info(line)

        send_email(RECEPIENTS["dayly"], body_lines)

    def check_hourly():
        table = '//home/avia/logs/avia-redir-log/%s' % datetime.today().strftime("%Y-%m-%d")

        tmp_table = yt.create_temp_table()

        log.info("Run map")
        yt.run_map(
            convert_eventtime,
            source_table=table,
            destination_table=tmp_table,
            format=yt.DsvFormat(),
        )

        log.info("Run sort")
        yt.run_sort(
            tmp_table,
            tmp_table,
            sort_by=['eventtime']
        )

        log.info("Run reduce")
        yt.run_reduce(
            aggregate_eventtime,
            tmp_table,
            tmp_table,
            format=yt.DsvFormat(),
            reduce_by=['eventtime'],
        )

        log.info("Calc last eventtime")
        last_eventtime = None
        for record in yt.read_table(tmp_table, format=yt.DsvFormat(), raw=False):
            eventtime = record["eventtime"]
            eventtime = datetime.strptime(eventtime, "%Y-%m-%d %H:%M")

            if not last_eventtime or eventtime > last_eventtime:
                last_eventtime = eventtime

        now = datetime.now()

        if not last_eventtime:
            err_msg = "No eventime in logs"
        elif last_eventtime > datetime.now():
            err_msg = "Eventtime in future: %s" % eventtime.strftime('%Y-%m-%d %H:%M')
        elif (last_eventtime + HOURLY_MAX_TIMEDELTA) < now:
            err_msg = "Last eventtime: %s" % eventtime.strftime('%Y-%m-%d %H:%M')
        else:
            err_msg = None

        if err_msg:
            log.info("ERORR: %s", err_msg)
            body_lines = ["WARNING!", "", err_msg]
            send_email(RECEPIENTS["hourly"], body_lines)

    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-t', '--type', default="dayly")

    options, args = optparser.parse_args()

    configure_wrapper(yt)

    if options.verbose:
        print_log_to_stdout(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    current_env = settings.ENVIRONMENT
    if current_env not in ALLOWED_ENVS:
        allowed_envs_str = ', '.join(ALLOWED_ENVS)
        log.info('Current ENVIRONMENT %s. Run only %s allowed.' % (current_env, allowed_envs_str))
        sys.exit()

    try:
        if options.type == "dayly":
            check_dayly()
        elif options.type == "hourly":
            check_hourly()
        else:
            log.error("Unknown type: %s", options.type)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')
