# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging

import requests
from lxml import etree
from django.conf import settings
from django.db import transaction

from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
from travel.avia.library.python.common.models.partner import Partner, RegionalizePartnerQueryRule
from travel.avia.library.python.common.utils.safe_xml_parser import safe_xml_fromstring

log = logging.getLogger(__name__)

URL = 'http://gate.ru-charter.ru/xml.asp'
CLIENT_ID = 545


def get_settlement(iata):
    settlement = None

    try:
        settlement = Settlement.objects.get(iata=iata)

    except Settlement.DoesNotExist:
        pass

    if not settlement:
        try:
            station = Station.get_by_code('iata', iata)
            settlement = station.settlement

        except Station.DoesNotExist:
            log.warning(u'Unknown IATA in chartex codes: %s' % iata)
            pass

    return settlement


def build_query_xml():
    data = etree.Element('data')
    request = etree.SubElement(data, 'request')

    request.set(str('type'), str('getRoutes'))

    client = etree.SubElement(data, 'client')
    etree.SubElement(client, 'clientId').text = str(CLIENT_ID)
    etree.SubElement(client, 'key').text = str(settings.CHARTEROK_KEY)

    return etree.tostring(
        data,
        encoding='utf-8',
        xml_declaration=True
    )


def get_data():
    r = requests.post(
        URL,
        headers={
            'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
        },
        data=build_query_xml()
    )

    return r.content


@transaction.atomic
def update_cache(xml):
    tree = safe_xml_fromstring(xml)

    partner = Partner.objects.get(code='charterok')
    partner.regionalizepartnerqueryrule_set.all().delete()

    for route in tree.xpath('//data/Routes/Route'):
        city_from_iata = route.attrib['city_from_iata']
        city_to_iata = route.attrib['city_to_iata']

        settlement_from = get_settlement(city_from_iata)
        settlement_to = get_settlement(city_to_iata)

        if settlement_from and settlement_to:
            regionalize_partner_query_rule, created = RegionalizePartnerQueryRule.objects.get_or_create(
                settlement_from=settlement_from,
                settlement_to=settlement_to,
                country_from=settlement_from.country,
                country_to=settlement_to.country,
                partner=partner
            )

            log_string_template = '%%s rule %s - %s: %s - %s' % (
                city_from_iata, city_to_iata,
                settlement_from.L_title(), settlement_to.L_title()
            )

            if created:
                log.debug(log_string_template % 'Add')
                regionalize_partner_query_rule.save()

            else:
                log.debug(log_string_template % 'Skip')

    query_rules_count = partner.regionalizepartnerqueryrule_set.count()
    log.info(u'Charterok query rules count: %s' % query_rules_count)


def _main():
    log.info(u'Start import')

    try:
        update_cache(
            get_data()
        )

    except Exception:
        log.exception('Import error:')

    log.info(u'End import')


def main():
    create_current_file_run_log()

    from optparse import OptionParser

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-m', '--check_maintenance', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout(log)

    _main()
