# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging

import requests

from django.db import transaction
from django.db.models import Q

from travel.avia.library.python.common.models.partner import Partner, RegionalizePartnerQueryRule
from travel.avia.library.python.common.models.geo import CodeSystem, Settlement, SettlementCode, StationCode
from travel.avia.library.python.common.utils.safe_xml_parser import safe_xml_fromstring

from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log

log = logging.getLogger(__name__)

CHARTEX_URL = 'http://xml.chartex.ru/gate/gate.asp'
CHARTEX_UID = '61500874'


def get_settlement(iata):
    settlement = None

    try:
        settlement = Settlement.objects.get(iata=iata)

    except Settlement.DoesNotExist:
        pass

    if not settlement:
        try:
            station = StationCode.objects.get(
                Q(system__code__iexact='iata') | Q(system__code__iexact='sirena'),
                code__iexact=iata
            )
            settlement = station.station.settlement

        except StationCode.DoesNotExist:
            pass

    return settlement


@transaction.atomic
def update_cache(chartex_routes):
    partner = Partner.objects.get(code='chartex')
    chartex_system = CodeSystem.objects.get(code='chartex')

    partner.regionalizepartnerqueryrule_set.all().delete()
    chartex_system.settlementcode_set.all().delete()

    chartex_cities = set()

    for route in chartex_routes:
        city_from_id, city_from_iata, city_to_id, city_to_iata = route

        chartex_cities.add((city_from_id, city_from_iata))
        chartex_cities.add((city_to_id, city_to_iata))

        settlement_from = get_settlement(city_from_iata)
        settlement_to = get_settlement(city_to_iata)

        if settlement_from and settlement_to:
            regionalize_partner_query_rule = RegionalizePartnerQueryRule(
                settlement_from=settlement_from,
                settlement_to=settlement_to,
                country_from=settlement_from.country,
                country_to=settlement_to.country,
                partner=partner
            )

            regionalize_partner_query_rule.save()

    if (8502, 'DWC') in chartex_cities and (5414, 'DXB') in chartex_cities:
        chartex_cities.remove((8502, 'DWC'))

    saved_settlement_ids = set()
    for idx, (city_id, city_iata) in enumerate(chartex_cities):
        settlement = get_settlement(city_iata)

        if not settlement:
            log.warning(u'Unknown IATA in chartex codes: %s' % city_iata)
            continue

        if settlement.id in saved_settlement_ids:
            log.warning(u'Cant save one settlement %r twice for city_id: %s, city_iata: %r', settlement, city_id, city_iata)
            continue
        saved_settlement_ids.add(settlement.id)

        settlement_code = SettlementCode(
            settlement=settlement,
            code=city_id,
            system=chartex_system
        )

        settlement_code.save()

    query_rules_count = partner.regionalizepartnerqueryrule_set.count()
    log.info(u'Chartex query rules count: %s' % query_rules_count)

    system_codes_count = chartex_system.settlementcode_set.count()
    log.info(u'Chartex settlements codes count: %s' % system_codes_count)


def get_data(params):
    params['UID'] = CHARTEX_UID

    r = requests.post(
        CHARTEX_URL,
        data=params
    )

    log.debug(u'Run get_data %s\n%s', params, r.text)

    return r.content


def parse_countries(xml):
    tree = safe_xml_fromstring(xml)

    return [
        country.get('id')
        for country in tree.xpath('country')
    ]


def parse_cities(xml):
    tree = safe_xml_fromstring(xml)

    return [
        (city.get('id'), city.get('CityCode'))
        for city in tree.xpath('city')
    ]


def parse_countries_cities(xml):
    return parse_cities(xml)


def get_routes():
    chartex_routes = []

    departures_xml = get_data({'request': 'cityfrom'})

    departure_cities = parse_countries_cities(departures_xml)

    for city_from_id, city_from_iata in departure_cities:
        arrival_countries_xml = get_data({
            'request': 'countryto',
            'CityID': city_from_id
        })

        arrival_countries = parse_countries(arrival_countries_xml)

        for country_from_id in arrival_countries:
            arrival_cities_xml = get_data({
                'request': 'cityto',
                'CityID': city_from_id,
                'CID': country_from_id,
            })

            arrival_cities = parse_cities(arrival_cities_xml)

            for city_to_id, city_to_iata in arrival_cities:
                chartex_routes.append((
                    city_from_id,
                    city_from_iata,
                    city_to_id,
                    city_to_iata
                ))

    log.info(u'Parsed chartex routes: %d', len(chartex_routes))

    return chartex_routes


def _main():
    log.info(u'Start import')

    try:
        update_cache(get_routes())

    except Exception:
        log.exception(u'Chartex fetch error')

    log.info(u'Done')


def main():
    from optparse import OptionParser

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    create_current_file_run_log()

    _main()
