# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
import requests
import json
from urlparse import urlparse

from django.db import transaction
from django.conf import settings
from django.core.mail.message import EmailMultiAlternatives

from travel.avia.library.python.common.models.partner import DohopIndex, DohopVendor, Partner
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log

log = logging.getLogger(__name__)

RECIPIENTS = ['avia-info@yandex-team.ru']

DOHOP = Partner.objects.get(code='dohop')


def shorten(url):
    parsed = urlparse(url)
    if not parsed.scheme and not parsed.netloc:
        parsed = urlparse('http://' + url)
    fqdn = parsed.netloc
    return fqdn[4:] if fqdn.startswith('www.') else fqdn


def _vendor_site_url(vinfo):
    url = vinfo.get('website', '')
    return shorten(url) if url else ''


@transaction.atomic
def update_vendors(vendors_by_index):
    return sorted(_update_vendors(vendors_by_index))


def _update_vendors(vendors_by_index):
    for index_code, vendors in vendors_by_index.items():
        index_obj, _created_index = DohopIndex.objects.get_or_create(code=index_code)
        for vinfo in vendors:
            try:
                vendor_db, is_created_vendor = DohopVendor.objects.get_or_create(
                    dohop_id=int(vinfo['id']),
                )

                log.info(
                    '%s vendor with id: %s, set name: %s',
                    'Add new' if is_created_vendor else 'Update',
                    vinfo['id'],
                    vinfo['name'],
                )

                vendor_db.title = vinfo['name']

                vendor_db.dohop_code = vinfo['name']

                vendor_db.site_url = vendor_db.site_url or _vendor_site_url(vinfo)

                vendor_db.partner = DOHOP
                vendor_db.billing_datasource_id_dev = DOHOP.billing_datasource_id_dev
                vendor_db.billing_datasource_id_testing = DOHOP.billing_datasource_id_testing
                vendor_db.billing_datasource_id_production = DOHOP.billing_datasource_id_production
                vendor_db.click_price = DOHOP.click_price
                vendor_db.t_type = DOHOP.t_type

                vendor_db.save()

                index_info_obj, _created_index_info = vendor_db.index_infos.get_or_create(index=index_obj)
                index_info_obj.vendor_info = json.dumps(vinfo, indent=4)
                index_info_obj.residences = ', '.join(vinfo['res'])
                if 'lang' in vinfo:
                    index_info_obj.langs = ', '.join(vinfo['lang'])
                index_info_obj.save()

                if is_created_vendor:
                    yield '{}:{}'.format(index_code, vinfo['name'])

            except Exception:
                log.exception('Error storing')


def send_email(receipents, new_vendors):
    subject = u'Новые вендоры Dohop'
    body = u'Добавлены новые вендоры:\n\n %s' % (',\n '.join(new_vendors))

    mail = EmailMultiAlternatives(
        subject=subject,
        body=body,
        from_email=settings.SERVER_EMAIL,
        to=receipents,
    )

    mail.send()


def fetch_vendors_from_index(index):
    url = 'https://api.dohop.com/api/v1/vendor-info/{}'.format(index)
    log.info('Get %s', url)
    r = requests.get(url, timeout=60)
    if not r.ok:
        raise Exception('Fetch error: [%s] %s %r' % (r.status_code, r.request.url, r.content))
    return r.json()['vendor-info']


def _main(options):
    log.info('Start')

    vendors_by_index = {
        index_code: fetch_vendors_from_index(index_code)
        for index_code in ['yandex', 'yandexcom']
    }

    added_vendors = update_vendors(vendors_by_index)

    if added_vendors and not options.silence:
        log.info('Send email notification about new vendors: %s', ', '.join(added_vendors))
        send_email(RECIPIENTS, added_vendors)

    log.info('Done')


def main():
    from optparse import OptionParser

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-s', '--silence', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    create_current_file_run_log()

    try:
        _main(options)
    except:
        log.exception('ERROR')
        raise
