# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import csv
import gzip
import io
import logging
from datetime import datetime
from optparse import OptionParser
from traceback import format_exc

from cachetools.func import lru_cache
import requests
from django.db import transaction
from django.conf import settings

from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.models.partner import Partner, RegionalizePartnerQueryRule
from travel.avia.library.python.proxy_pool.deploy_proxy_pool import DeployProxyPool

from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log

log = logging.getLogger(__name__)

URL = 'http://pub.pobeda.aero/schedule/Schedule.csv.gz'

proxy_pool = DeployProxyPool(
    environment=settings.ENVIRONMENT,
    datacenter=settings.CURRENT_DC,
    login=settings.PARTNER_PROXY_LOGIN,
    password=settings.PARTNER_PROXY_PASSWORD,
    logger=log,
)

http_proxy = proxy_pool.get_proxy().get_http_uri()


@lru_cache(maxsize=None)
def get_settlement(iata):
    settlement = None

    try:
        settlement = Settlement.objects.get(iata=iata)

    except Settlement.DoesNotExist:
        pass

    if not settlement:
        try:
            station = Station.get_by_code('iata', iata)
            settlement = station.settlement

        except Station.DoesNotExist:
            log.warning(u'Unknown IATA in pobeda codes: %s', iata)

    return settlement


def get_data():
    log.info('Get %s', URL)
    r = requests.get(URL, timeout=60, proxies={'http': http_proxy})
    if not r.ok:
        raise Exception('Fetch error: [%s] %s %r' % (r.status_code, r.request.url, r.content))

    return r.content


@transaction.atomic
def update_cache(data):
    partner = Partner.objects.get(code='pobeda')
    partner.regionalizepartnerqueryrule_set.all().delete()

    rules = []
    with gzip.GzipFile(fileobj=io.BytesIO(data)) as f:
        csv_reader = csv.DictReader(f, delimiter=';')

        for row in csv_reader:
            station_from_iata = row['DepartureStation']
            station_to_iata = row['ArrivalStation']
            start_date = datetime.strptime(row['BeginDate'], '%d.%m.%y').date()
            end_date = datetime.strptime(row['EndDate'], '%d.%m.%y').date()
            week_days = row['DepartureDOW']
            settlement_from = get_settlement(station_from_iata)
            settlement_to = get_settlement(station_to_iata)

            if not settlement_from or not settlement_to:
                continue

            rules.append(
                RegionalizePartnerQueryRule(
                    settlement_from=settlement_from,
                    settlement_to=settlement_to,
                    country_from=settlement_from.country,
                    country_to=settlement_to.country,
                    start_date=start_date,
                    end_date=end_date,
                    week_days=week_days,
                    partner=partner
                )
            )

    RegionalizePartnerQueryRule.objects.bulk_create(rules)
    query_rules_count = partner.regionalizepartnerqueryrule_set.count()
    log.info(u'Pobeda query rules count: %s', query_rules_count)


def _main():
    log.info(u'Start import')

    try:
        update_cache(
            get_data()
        )

    except Exception:
        log.exception(u'Import error: %s', format_exc())

    log.info(u'End import')


def main():
    create_current_file_run_log()

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout(log)

    _main()
