# -*- encoding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging
import json
import os
import zlib
from collections import defaultdict
from contextlib import closing
from ftplib import FTP
from itertools import chain, product
from optparse import OptionParser
from StringIO import StringIO
from traceback import format_exc

from django.conf import settings
from django.core.cache import get_cache

from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.settings import dc_cache
from travel.avia.library.python.common.utils.threadutils import run_in_thread
from travel.avia.admin.lib.logs import print_log_to_stdout, create_current_file_run_log
from travel.avia.library.python.common.lib.mail import mail_message
from travel.avia.admin.lib.unicode_csv import UnicodeDictReader


script_name = os.path.basename(__file__)

log = logging.getLogger(__name__)

HOST = 'fares.pobeda.aero'
USER = 'yandex'
PASSWORD = 'diZuR%yF'
FTP_FILE = 'AvailableFares.csv'

POBEDA_MEMCACHE_KEY = 'pobeda_v3'

POBEDA_FILE_SAVETO = '/tmp/pobeda_ftp_data'


def download_file(filename, host, user, password):
    pobeda_file_dump = getattr(settings, 'POBEDA_FILE_DUMP', None)
    if pobeda_file_dump:
        log.warning('Using dump file: %r', pobeda_file_dump)

        with open(pobeda_file_dump, 'rb') as f:
            return f.read()

    with closing(StringIO()) as f:
        ftp = FTP(host, timeout=60 * 30)

        with closing(ftp) as ftp:
            ftp.login(user, password)
            ftp.retrbinary('RETR ' + filename, f.write, 1024)

        file_data = f.getvalue()

    try:
        with open(POBEDA_FILE_SAVETO, 'wb') as f:
            f.write(file_data)

    except Exception:
        log.exception('Error saving file to %r', POBEDA_FILE_SAVETO)

    return file_data


def get_pobeda_json():
    csv = download_file(FTP_FILE, HOST, USER, PASSWORD)
    reader = UnicodeDictReader(StringIO(csv), delimiter=';')

    return list(reader)


def get_settlements_by_iata_codes(iata_codes):
    settlements = Settlement.objects. \
        filter(hidden=False, iata__in=iata_codes).distinct()

    return {s.iata: s for s in settlements}


def get_stations_by_iata_codes(iata_codes):
    ids_and_codes = set(
        Station.objects.filter(
            hidden=False,
            t_type__code='plane',
            code_set__system__code='iata',
            code_set__code__in=iata_codes
        ).values_list('id', 'code_set__code').distinct()
    )

    stations = Station.objects.in_bulk([id for id, code in ids_and_codes])

    return {code: stations[id] for id, code in ids_and_codes}


def get_extended_iatas(iata_codes):
    extended = defaultdict(set)

    for code in iata_codes:
        extended[code].add(code)

    settlements_by_codes = get_settlements_by_iata_codes(iata_codes)
    codes_by_settlements_ids = {s.id: code
                                for code, s in settlements_by_codes.items()}

    # Коды городов
    for code, settlement in settlements_by_codes.items():
        extended[code].add(settlement.iata)
        # extended[code].add(settlement.sirena_id)

    # Коды аэропортов в городах
    settlements_ids_codes = (
        Settlement.objects.filter(
            id__in=[s.id for s in settlements_by_codes.values()],
            station__hidden=False,
            station__t_type__code='plane',
            station__code_set__system__code__in=[
                'iata',
                # 'sirena'
            ]
        )
        .values_list('id', 'station__code_set__code')
        .distinct()
    )

    for sid, code in settlements_ids_codes:
        extended[codes_by_settlements_ids[sid]].add(code)

    # Коды аэропортов
    stations_by_codes = get_stations_by_iata_codes(iata_codes)

    Station.code_manager.precache(systems=[
        'iata',
        # 'sirena'
    ])

    for code, station in stations_by_codes.items():
        codes = Station.code_manager.getcodes(station)
        # extended[code].update(codes.values())

        for system in [
            'iata',
            # 'sirena',
        ]:
            extended[code].add(codes[system])

    # Коды городов аэропортов
    for code, station in stations_by_codes.items():
        extended[code].add(station.settlement.iata)

    return {k: tuple(sorted(filter(None, v))) for k, v in extended.items()}


def gen_pobeda_variants(data):
    iata_directions = set([(d['iata_from'], d['iata_to']) for d in data])
    iata_codes = set(chain.from_iterable(iata_directions))
    extended = get_extended_iatas(iata_codes)

    for v in data:
        iatas_from = extended[v['iata_from']]
        iatas_to = extended[v['iata_to']]

        for iata_from, iata_to in product(iatas_from, iatas_to):
            yield VariantWrapper(v, iata_from, iata_to, v['FlightDate'])


class VariantWrapper(object):
    def __init__(self, data, iata_from, iata_to, flight_date):
        self.data = data
        self.iata_from = iata_from
        self.iata_to = iata_to
        self.flight_date = flight_date

    @property
    def key(self):
        return '%s_%s_%s' % (self.iata_from, self.iata_to, self.flight_date)


def host_cache_set_multi(host, mapping, timeout):
    cache = get_cache(
        'django.core.cache.backends.memcached.MemcachedCache',
        **dc_cache([host])
    )

    try:
        cache.set_many(mapping, timeout)

    except Exception:
        log.exception('Storing multikeys to cache error')

    finally:
        cache.close()


def cluster_cache_set_multi(mapping, timeout):
    log.debug(u'Store keys: %s', len(mapping))

    if not settings.DAEMON_CACHE_HOSTS:
        return

    workers = [
        run_in_thread(host_cache_set_multi, (host, mapping, timeout))
        for host in settings.DAEMON_CACHE_HOSTS
    ]

    for worker in workers:
        worker.join()


def pobeda_direction_memcache_key(direction_key):
    return '%s_%s' % (POBEDA_MEMCACHE_KEY, direction_key)


def _main():
    data = get_pobeda_json()

    log.info('DAEMON_CACHE_HOSTS: %r', settings.DAEMON_CACHE_HOSTS)

    for v in data:
        v['iata_from'] = v['FlightRoute'][:3]
        v['iata_to'] = v['FlightRoute'][-3:]

        log.debug('Variant: %r', v)

    by_key = defaultdict(list)

    for v in gen_pobeda_variants(data):
        by_key[v.key].append(v.data)

    keys_to_save = {
        pobeda_direction_memcache_key(key): zlib.compress(json.dumps(variants))
        for key, variants in by_key.items()
    }

    try:
        # Запоминаем на 10 минут
        cluster_cache_set_multi(keys_to_save, 60 * 10)

    except Exception:
        log.exception('Store data error')

    log.info('Keys count: %s', len(by_key))


def mail_error(e, message):
    if settings.AVIA_PARTNER_FETCH_SUBJECT:
        mail_message(
            'Pobeda [%s]: %r' % (settings.AVIA_PARTNER_FETCH_SUBJECT, e),
            message=message,
            recipients=[('Avia partner fetch mail list',
                         settings.AVIA_PARTNER_FETCH_MAIL)]
        )


usage = u'Usage: python %prog [options]'


def main():
    optparser = OptionParser(usage=usage, description=__doc__)
    optparser.add_option('-v', '--verbose', action='store_true',
                         help=u'выводить лог на экран')
    options, args = optparser.parse_args()

    if options.verbose:
        print_log_to_stdout()

    create_current_file_run_log()

    try:
        _main()

    except Exception as e:
        log.exception(u'Fetch error')

        try:
            mail_error(e, format_exc())

        except Exception:
            log.exception('Mail error')
