# -*- coding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import argparse
import csv
import io
import logging
from collections import defaultdict
from itertools import product

import requests
from django.db import transaction
from django.db.models import Q

from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.yav import get_secret_value
from travel.avia.library.python.common.models.geo import Settlement, StationCode
from travel.avia.library.python.common.models.partner import Partner, RegionalizePartnerQueryRule


log = logging.getLogger(__name__)

FETCH_FROM_PARTNER_TIMEOUT = 60


class SettlementCache(object):
    def __init__(self, memo):
        self._memo = memo

    @classmethod
    def precached(cls):
        memo = defaultdict(list)
        for station_code in StationCode.objects.select_related('station__settlement').filter(
            Q(system__code__iexact='iata') | Q(system__code__iexact='sirena'),
            station__settlement__isnull=False
        ):
            memo[station_code.code.lower()].append(station_code.station.settlement)

        for settlement in Settlement.objects.filter(iata__isnull=False):
            memo[settlement.iata.lower()].append(settlement)

        for settlement in Settlement.objects.filter(sirena_id__isnull=False):
            memo[settlement.sirena_id.lower()].append(settlement)

        return cls(dict(memo))

    def settlement(self, code):
        try:
            return self._memo[code.lower()][0]
        except KeyError:
            raise Exception(
                'Settlement for code {!r} is not found'.format(code)
            )

    def settlements(self, code):
        try:
            return self._memo[code.lower()]
        except KeyError:
            raise Exception(
                'Settlements for code {!r} are not found'.format(code)
            )


class RuleSource(object):
    def __init__(self, partner_code, url, basic_auth=None):
        self._partner_code = partner_code
        self._url = url
        self._basic_auth = basic_auth

    def whitelist(self, settlement_cache):
        response = requests.get(
            self._url,
            auth=self._basic_auth,
            headers={'user-agent': 'Yandex Avia Bot'},
            verify=False,
            timeout=FETCH_FROM_PARTNER_TIMEOUT,
        )
        response.raise_for_status()
        content = response.content
        log.debug('%s content %s', self, content)
        return PartnerWhitelist.from_raw(
            partner=Partner.objects.get(code=self._partner_code),
            raw_whitelist=content,
            settlement_cache=settlement_cache,
        )

    def __str__(self):
        return 'RuleSource({}, {}, basic_auth={})'.format(
            self._partner_code, self._url, bool(self._basic_auth)
        )


class QueryRule(object):
    def __init__(self, settlement_from, settlement_to):
        self.settlement_from = settlement_from
        self.settlement_to = settlement_to

    @classmethod
    def from_string(cls, string, settlement_cache):
        code_from, code_to = string.strip().strip('"').split(';')
        for settlement_from, settlement_to in product(
            settlement_cache.settlements(code_from.strip()),
            settlement_cache.settlements(code_to.strip())
        ):
            yield cls(
                settlement_from=settlement_from,
                settlement_to=settlement_to,
            )

    def reversed(self):
        return QueryRule(
            settlement_to=self.settlement_from,
            settlement_from=self.settlement_to
        )

    def as_model(self, partner):
        return RegionalizePartnerQueryRule(
            partner=partner,
            settlement_from=self.settlement_from,
            country_from=self.settlement_from.country,
            settlement_to=self.settlement_to,
            country_to=self.settlement_to.country,
        )


class PartnerWhitelist(object):
    def __init__(self, partner, rules):
        self.partner = partner
        self.rules = rules

    @classmethod
    def from_raw(cls, partner, raw_whitelist, settlement_cache):
        rules = []
        for [raw_rule] in csv.reader(io.BytesIO(raw_whitelist)):
            try:
                for rule in QueryRule.from_string(raw_rule, settlement_cache):
                    rules.append(rule)
                    rules.append(rule.reversed())

            except Exception as exc:
                log.exception('Rule creation error (%r): %r', raw_rule, exc)

        return cls(partner, rules)

    @transaction.atomic
    def store(self):
        RegionalizePartnerQueryRule.objects.filter(
            partner=self.partner
        ).delete()

        unique_models = {
            (
                model.partner_id,
                model.settlement_from_id,
                model.settlement_to_id,
            ):
               model
            for model in (
                rule.as_model(self.partner) for rule in self.rules
            )
        }.values()

        RegionalizePartnerQueryRule.objects.bulk_create(
            unique_models, batch_size=2000
        )


def _main(args):
    log.info(u'Started the import')

    settlement_cache = SettlementCache.precached()

    log.info(u'Precached the settlements')

    sources = (
        RuleSource(
            'chartertickets',
            'https://chartertickets.com.ua/directions.txt',
        ),
        RuleSource(
            's_seven',
            'https://www.s7.ru/directions.txt',
            basic_auth=('avia-info', get_secret_value(args.s7_secret_id, args.s7_secret_key)),
        ),
        RuleSource(
            'chabooka',
            'http://api.chabooka.ru/cities.php?partner=yandex',
        ),
        RuleSource(
            'ufs_plane',
            'https://static.ufs-online.ru/yandex/Directions_Yandex.csv',
        ),
    )
    sources_with_errors = []
    for source in sources:
        try:
            source.whitelist(settlement_cache).store()
        except Exception as exc:
            sources_with_errors.append(source)
            log.exception(u'Import error for %s: %r', source, exc)
        else:
            log.info(u'Rules from %s are imported successfully', source)

    log.info(u'Done')

    if sources_with_errors:
        raise Exception('Error in fetch \n%s', '\n'.join(map(str, sources_with_errors)))


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    parser.add_argument('--s7-secret-id', default='sec-01f8x5abj7xayx8gs2kyx2s61x')
    parser.add_argument('--s7-secret-key', default='password')
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(log)

    create_current_file_run_log()

    _main(args)
