import travel.avia.admin.init_project  # noqa

import logging

from django.conf import settings
from django.db import transaction
from typing import List

from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.models.partner import RegionalizePartnerQueryRule
from travel.avia.library.python.proxy_pool.deploy_proxy_pool import DeployProxyPool


class Fetcher(object):
    def __init__(self, logger):
        # type: (logging.Logger) -> None
        self._logger = logger
        self._settlement_by_iata = {}
        self._partner = None

    def fetch_rules_and_update(self):
        self._logger.info('Start')

        rules = self._fetch_rules()
        self._logger.info('Fetched %d rules', len(rules))

        self._update_rules(rules)

        self._logger.info('Done')

    def _get_partner_proxy(self):
        proxy_pool = DeployProxyPool(
            environment=settings.ENVIRONMENT,
            datacenter=settings.CURRENT_DC,
            login=settings.PARTNER_PROXY_LOGIN,
            password=settings.PARTNER_PROXY_PASSWORD,
            logger=self._logger,
        )

        return proxy_pool.get_proxy()

    def _get_settlement_by_iata(self, iata):
        # type: (str) -> Settlement
        if iata in self._settlement_by_iata:
            return self._settlement_by_iata[iata]

        settlement = None

        try:
            settlement = Settlement.objects.get(iata=iata)
        except Settlement.DoesNotExist:
            pass

        if not settlement:
            try:
                station = Station.get_by_code('iata', iata)
                settlement = station.settlement
            except Station.DoesNotExist:
                self._logger.warning('Unknown IATA: %s', iata)

        self._settlement_by_iata[iata] = settlement

        return settlement

    def _fetch_rules(self):
        # type: () -> List[RegionalizePartnerQueryRule]
        return []

    @transaction.atomic
    def _update_rules(self, rules):
        # type: (List[RegionalizePartnerQueryRule]) -> None
        self._logger.info('Remove old rules for "%s" ...', self._partner)
        RegionalizePartnerQueryRule.objects.filter(partner=self._partner).delete()

        self._logger.info('Save %d rules for "%s" ...', len(rules), self._partner)
        RegionalizePartnerQueryRule.objects.bulk_create(rules)
