import travel.avia.admin.init_project  # noqa

import argparse
import csv
import ftplib
import logging
from datetime import datetime

from typing import List

from travel.avia.admin.avia_scripts.fetch_regionalization import Fetcher
from travel.avia.admin.lib.yav import get_secret
from travel.avia.library.python.common.models.partner import Partner, RegionalizePartnerQueryRule


class AzimuthFetcher(Fetcher):
    FTP_HOST = 'ftp.tais.ru'

    def __init__(self, logger, ftp_user, ftp_password):
        # type: (logging.Logger, str, str) -> None
        super(AzimuthFetcher, self).__init__(logger)

        self._partner = Partner.objects.get(code='azimuth')
        self._ftp_user = ftp_user
        self._ftp_password = ftp_password

    def _fetch_rules(self):
        # type: () -> List[RegionalizePartnerQueryRule]
        proxy = self._get_partner_proxy()

        ftp = ftplib.FTP(timeout=20)
        ftp.connect(
            host=proxy.get_ftp_host(),
            port=proxy.FTP_PORT,
        )
        ftp.login(
            user=proxy.get_ftp_user(self._ftp_user, self.FTP_HOST),
            passwd=self._ftp_password,
        )

        blocks = []
        ftp.set_pasv(True)

        files = []
        ftp.dir(lambda s: files.append(s.split(' ')[-1]))
        files.sort()

        self._logger.info('Fetch rules from "%s"', files[-1])
        ftp.retrbinary('RETR {}'.format(files[-1]), callback=blocks.append)

        data = ''.join(blocks)

        rules = []

        reader = csv.DictReader(data.splitlines())
        for row in reader:
            settlement_from = self._get_settlement_by_iata(row['iata_from'])
            settlement_to = self._get_settlement_by_iata(row['iata_to'])
            rules.append(
                RegionalizePartnerQueryRule(
                    settlement_from=settlement_from,
                    settlement_to=settlement_to,
                    country_from=settlement_from.country,
                    country_to=settlement_to.country,
                    start_date=datetime.strptime(row['date_from'], '%Y-%m-%d').date() if row['date_from'] else None,
                    end_date=datetime.strptime(row['date_to'], '%Y-%m-%d').date() if row['date_to'] else None,
                    week_days=row['weekdays'],
                    partner=self._partner,
                )
            )

        return rules


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--yav-secret-id', required=True)
    args = parser.parse_args()

    logger = logging.getLogger(__name__)

    secret = get_secret(args.yav_secret_id)

    fetcher = AzimuthFetcher(
        logger=logger,
        ftp_user=secret['regionalization-login'],
        ftp_password=secret['regionalization-password'],
    )
    fetcher.fetch_rules_and_update()
