import travel.avia.admin.init_project  # noqa

import logging
import requests
from datetime import datetime

from typing import List

from travel.avia.admin.avia_scripts.fetch_regionalization import Fetcher
from travel.avia.library.python.common.models.partner import Partner, RegionalizePartnerQueryRule


class SmartaviaFetcher(Fetcher):
    SCHEDULE_URL = 'https://ya1-api.flysmartavia.com/api/avia/schedule'
    CONNECT_TIMEOUT = 60
    READ_TIMEOUT = 60

    def __init__(self, logger):
        # type: (logging.Logger) -> None
        super(SmartaviaFetcher, self).__init__(logger)
        self._partner = Partner.objects.get(code='smartavia')

    def _fetch_rules(self):
        # type: () -> List[RegionalizePartnerQueryRule]
        proxies = self._get_partner_proxy().get_requests_proxies()
        fetching_params = {
            'airline': '5N',
        }

        response = requests.get(
            self.SCHEDULE_URL,
            params=fetching_params,
            timeout=(self.CONNECT_TIMEOUT, self.READ_TIMEOUT),
            proxies=proxies,
        )
        response.raise_for_status()
        content = response.json()

        rules = []

        for elem in content['data']['schedule']:
            settlement_from = self._get_settlement_by_iata(elem['from']['airport']['code'])
            settlement_to = self._get_settlement_by_iata(elem['to']['airport']['code'])
            for period in elem['periods']:
                rules.append(
                    RegionalizePartnerQueryRule(
                        settlement_from=settlement_from,
                        settlement_to=settlement_to,
                        country_from=settlement_from.country,
                        country_to=settlement_to.country,
                        start_date=datetime.strptime(period['navigation_start'], '%Y-%m-%d').date(),
                        end_date=datetime.strptime(period['navigation_end'], '%Y-%m-%d').date(),
                        week_days=''.join(str(x) for x in period['weekdays']),
                        partner=self._partner,
                    )
                )

        self._logger.info('Fetched %d regionalization rules', len(rules))
        return rules


def main():
    logger = logging.getLogger(__name__)

    fetcher = SmartaviaFetcher(logger=logger)
    fetcher.fetch_rules_and_update()
