import travel.avia.admin.init_project  # noqa

import csv
import logging
import requests
from typing import List

from travel.avia.admin.avia_scripts.fetch_regionalization import Fetcher
from travel.avia.library.python.common.models.partner import Partner, RegionalizePartnerQueryRule


# Example response
# MOW;LED;1,2,3,4,5,6,7
# MOW;AER;1,2,3,4,5,6,7
# MOW;MRV;1,2,3,4,5,6,7
# PKC;VVO;1,2,3,4,5,6,7

class TutuFetcher(Fetcher):
    SCHEDULE_URL = 'https://yapi.tutu.ru/restrictions'
    CONNECT_TIMEOUT = 60
    READ_TIMEOUT = 60

    def __init__(self, logger):
        # type: (logging.Logger) -> None
        super(TutuFetcher, self).__init__(logger)
        self._partner = Partner.objects.get(code='tutu')

    def _fetch_rules(self):
        # type: () -> List[RegionalizePartnerQueryRule]
        proxies = self._get_partner_proxy().get_requests_proxies()

        response = requests.get(
            self.SCHEDULE_URL,
            timeout=(self.CONNECT_TIMEOUT, self.READ_TIMEOUT),
            proxies=proxies,
        )
        response.raise_for_status()
        content = response.text

        rules = []
        errors = []
        r = csv.reader(content.splitlines(), delimiter=';')
        for row in r:
            if not row:
                continue
            try:
                settlement_from = self._get_settlement_by_iata(row[0])
                settlement_to = self._get_settlement_by_iata(row[1])
                rules.append(
                    RegionalizePartnerQueryRule(
                        settlement_from=settlement_from,
                        settlement_to=settlement_to,
                        country_from=settlement_from.country,
                        country_to=settlement_to.country,
                        week_days=row[2].replace(',', ''),
                        partner=self._partner,
                    )
                )
            except Exception as exc:
                errors.append(str(exc))

        if len(errors) > 0:
            raise Exception('Errors: %r', '\n'.join(errors))

        self._logger.info('Fetched %d regionalization rules', len(rules))
        return rules


def main():
    logger = logging.getLogger(__name__)

    fetcher = TutuFetcher(logger=logger)
    fetcher.fetch_rules_and_update()
