# -*- coding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging

import requests

from django.db import transaction
from django.conf import settings

from travel.avia.admin.www.models.geo import Settlement
from travel.avia.admin.avia.models import TourDeparture, TourDestination
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log

log = logging.getLogger(__name__)


TOURS_BASE_URL = 'http://%s:%d' % (settings.TOURS_HOST, settings.TOURS_PORT)
REQUEST_TIMEOUT = 1.0


@transaction.atomic
def update(departure, destination):
    TourDeparture.objects.all().delete()
    TourDestination.objects.all().delete()

    bulk = [TourDeparture(geo_id=s._geo_id, settlement=s) for s in departure]
    TourDeparture.objects.bulk_create(bulk, batch_size=1000)

    bulk = [TourDestination(geo_id=s._geo_id, settlement=s) for s in destination]
    TourDestination.objects.bulk_create(bulk, batch_size=1000)

    log.info('created %d departure settlements', len(departure))
    log.info('created %d destination settlements', len(destination))


def get_departure():
    r = requests.get('%s/api/1.x/reference/departures' % TOURS_BASE_URL, timeout=REQUEST_TIMEOUT)
    r.raise_for_status()
    geo_ids = [i[0] for i in r.json()['data']]

    settlements = Settlement.objects.filter(_geo_id__in=geo_ids)

    delta = len(geo_ids) - len(settlements)
    if delta:
        log.info('skipped %d departure settlements without link to model', delta)

    return settlements


def get_destination():
    r = requests.get('%s/api/1.x/reference/destinations' % TOURS_BASE_URL, timeout=REQUEST_TIMEOUT)
    r.raise_for_status()
    geo_ids = [i[0] for i in r.json()['data']]

    settlements = Settlement.objects.filter(_geo_id__in=geo_ids)

    delta = len(geo_ids) - len(settlements)
    if delta:
        log.info('skipped %d destination settlements without link to model', delta)

    return settlements


def _main():
    log.info(u'Start import')

    try:
        update(get_departure(), get_destination())

    except Exception:
        log.exception(u'fetch error')

    log.info(u'Done')


def main():
    from optparse import OptionParser

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    create_current_file_run_log()

    _main()
