# -*- encoding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
import requests
import sys
import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger
from collections import defaultdict
from datetime import date, datetime, timedelta
from optparse import OptionParser
from threading import Thread
from time import sleep
from Queue import Queue

from django.conf import settings
from django.db import transaction
from django.http import QueryDict

from travel.avia.library.python.avia_data.libs.currency.rates.caching import CurrencyRatesCache
from travel.avia.library.python.avia_data.models import AviaDirectionNational, MinPrice
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.yt_helpers import configure_wrapper
from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.partner import DohopVendor
from travel.avia.library.python.common.models.tariffs import Setting

log = logging.getLogger(__name__)

MAX_THREADS = 10
FAKE_EVENTTIME = datetime(2000, 1, 1)

API_ROOT = 'https://yandexapi.dohop.com'
API_URL = '%s/api/v1' % API_ROOT
INIT_ROOT = '%s/search' % API_URL
POOL_ROOT = '%s/poll' % API_URL
LINK_ROOT = '%s/transfer' % API_URL
MAX_REQ_TIME = 60
RESIDENCIES_MAP = {'ru': 'RU', 'ua': 'UA', 'tr': 'TR', 'com': 'GB'}
LANGUAGES_MAP = {'ru': 'ru', 'ua': 'uk', 'tr': 'tr', 'com': 'en'}
SETTING_CODE = 'AVIA_COM_PRICE_LMT'
SETTING_DEFAULTS = {
    'type': 'str',
    'name': u'Время последнего обновления турецких цен',
    'value': ''
}
NATIONAL_VERSION = 'com'
NATIONAL_CURRENCY = Currency.objects.get(code=settings.AVIA_NATIONAL_CURRENCIES[NATIONAL_VERSION])
DOHOP_VENDORS = [int(v.dohop_id) for v in DohopVendor.get_actual(national_version=NATIONAL_VERSION)]

DEPARTURES = [
    10466,  # Амстердам
    10426,  # Дублин
    177,    # Берлин
    10393,  # Лондон
    10511,  # Прага
    10467,  # Осло
    11474,  # Рига
    10493,  # Хельсинки
    10519,  # Стокгольм
    10445,  # Рим
    213,    # Москва
    54,     # Екатеринбург
]


def get_min_price(point_from, point_to, date_forward, date_backward, proxies):
    start_time = datetime.now()
    min_price = defaultdict(tuple)

    init_url = '/'.join(filter(None, [
        INIT_ROOT,
        'yandex',
        LANGUAGES_MAP.get(NATIONAL_VERSION, LANGUAGES_MAP['ru']),
        RESIDENCIES_MAP.get(NATIONAL_VERSION, RESIDENCIES_MAP['ru']),
        ','.join([a.iata for a in point_from.get_all_iata_airports() if a.t_type.code == 'plane']),
        ','.join([a.iata for a in point_to.get_all_iata_airports() if a.t_type.code == 'plane']),
        date_forward.strftime('%Y-%m-%d'),
        date_backward.strftime('%Y-%m-%d') if date_backward else None,
    ]))

    init_params = QueryDict('', mutable=True)
    init_params.update({
        'n_adults': 1,
        'faredata_cp_group_limit': 0,
        'include_vendors': ','.join(str(v) for v in DOHOP_VENDORS)
    })
    init_url += '?' + init_params.urlencode()

    try_count = 0
    success = False

    while try_count < 10 and not success:
        try:
            r = requests.get(
                init_url,
                timeout=30,
                proxies=proxies
            )
            success = True

        except:
            log.info('Retry: %s', init_url)
            try_count += 1
            continue

    init_json = r.json()

    error = init_json.get('error') or init_json.get('fatal_error')
    if error:
        log.error(error)
        return

    key, continuation = init_json['key'], init_json['continuation']

    results = IncrementalResult()

    while not results.is_done:
        # Уснем, что бы постоянный RPM не создавать
        sleep(1)

        if (datetime.now() - start_time).total_seconds() > 60:
            return

        pool_url = '%s/%s/%s' % (POOL_ROOT, key, continuation)

        try_count = 0
        success = False

        while try_count < 10 and not success:
            try:
                r = requests.get(
                    pool_url,
                    timeout=30,
                    proxies=proxies
                )
                success = True

            except:
                log.info('Retry: %s', pool_url)
                try_count += 1
                continue

        data_json = r.json()

        if 'fatal_error' in init_json:
            return

        continuation = data_json.get('continuation')

        results.extra_result(data_json)

    for fare_id, fare in results.fares.items():
        forward_numbers = parse_segments(fare['o'])
        backward_numbers = parse_segments(fare['h'])
        direct_flight = True if len(forward_numbers) == 1 and len(backward_numbers) == 1 else False

        for vendor_id, vendor_fare in fare['f'].items():
            price = float(vendor_fare['f'])
            vendor_currency = vendor_fare['c']

            conditions = [
                vendor_currency == NATIONAL_CURRENCY.code,
                vendor_currency in settings.AVIA_NATIONAL_VERSION_ALLOWED_FOREIGN_CURRENCIES[NATIONAL_VERSION]
            ]

            rates = CurrencyRatesCache.build(national_version=NATIONAL_VERSION).rates

            if not any(conditions):
                log.info('Skip %s', vendor_currency)
                continue

            try:
                if vendor_currency != NATIONAL_CURRENCY.code:
                    national_price = price * rates[vendor_currency.replace('RUB', 'RUR')] / rates[NATIONAL_CURRENCY.code]

                    log.info('Convert %s %s to %s %s', price, vendor_currency, national_price, NATIONAL_CURRENCY.code)
                    price = national_price

            except KeyError:
                continue

            if not min_price[direct_flight] or min_price[direct_flight][0] > price:
                routes = '%s/%s' % (','.join(forward_numbers), ','.join(backward_numbers))
                min_price[direct_flight] = (price, routes)

    return min_price


def next_weekday(d, weekday):
    days_ahead = weekday - d.weekday()

    if days_ahead <= 0:
        days_ahead += 7

    return d + timedelta(days_ahead)


def parse_segments(dohop_segments):
    flight_numbers = []

    for i in range(2, len(dohop_segments), 4):
        flight_number = dohop_segments[i]
        flight_numbers.append(flight_number)

    return flight_numbers


@transaction.atomic
def save_min_prices(top_direction, min_prices):
    min_prices_count = 0
    min_price_filter = {
        'national_version': NATIONAL_VERSION,
        'departure_settlement': top_direction.departure_settlement,
        'arrival_settlement': top_direction.arrival_settlement
    }

    MinPrice.objects.filter(**min_price_filter).delete()

    for price_info in min_prices:
        min_price, date_forward, date_backward = price_info

        for direct_flight, item in min_price.items():
            min_prices_count += 1
            price, routes = item
            mp, created = MinPrice.objects.get_or_create(
                departure_settlement=top_direction.departure_settlement,
                arrival_settlement=top_direction.arrival_settlement,
                date_forward=date_forward,
                date_backward=date_backward,
                passengers='1_0_0',
                national_version=NATIONAL_VERSION,
                direct_flight=direct_flight,
                routes=routes,
                defaults={
                    'price': price,
                    'currency': NATIONAL_CURRENCY,
                    'eventtime': datetime(2000, 1, 1),
                    'day_of_week': date_forward.weekday()
                }
            )

            if not created:
                mp.price = price
                mp.routes = routes
                mp.save()

    return min_prices_count


def process_top_direction(q):
    while True:
        min_prices = []
        task_number, tasks_total, top_direction, log, proxies = q.get()

        try:
            if task_number < MAX_THREADS:
                sleep(task_number * 1)

            log.info(u'*** %s / %s: Заполняем %s' % (
                task_number + 1, tasks_total, top_direction
            ))

            date_forward = next_weekday(date.today(), 3)

            for x in range(3):
                date_forward = date_forward + timedelta(days=x)
                date_backward = date_forward + timedelta(days=7)

                min_price = get_min_price(
                    top_direction.departure_settlement,
                    top_direction.arrival_settlement,
                    date_forward,
                    date_backward,
                    proxies
                )

                if min_price:
                    min_prices.append((min_price, date_forward, date_backward))

            min_prices_count = save_min_prices(top_direction, min_prices)

            log.info(u'*** %s / %s: Заполнили %s: %s вариантов (%s - %s)' % (
                task_number + 1, tasks_total, top_direction, min_prices_count,
                top_direction.departure_settlement, top_direction.arrival_settlement
            ))

        except Exception as e:
            log.error('*** Error in %s: %s', top_direction, e)

        q.task_done()


def _main():
    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-d", "--days", dest="days", default=7)
    optparser.add_option("-s", "--skip_check", action="store_true")
    optparser.add_option("-n", "--number", dest="quantity", default=6)
    optparser.add_option("-p", "--proxy", dest="proxy", default=settings.YT_PROXY)
    optparser.add_option("--http_proxy", dest="http_proxy", default='')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    proxies={}

    if options.http_proxy:
        proxies = {
            'http': options.http_proxy,
        }

    log.info('*** Start')

    # Проверим список вендоров
    if not DOHOP_VENDORS:
        log.info('Empty vendor lists for %s', NATIONAL_VERSION)
        sys.exit()

    # Проверим дату последнего обновления
    setting, _created = Setting.objects.get_or_create(code=SETTING_CODE, defaults=SETTING_DEFAULTS)
    if setting.value:
        run_after = datetime.strptime(setting.value, '%Y-%m-%d %H:%M:%S') + timedelta(days=7)
        if datetime.now() <= run_after:
            log.info('Last run on %s, skip running', setting.value)
            sys.exit()

    configure_wrapper(yt)
    if options.proxy != settings.YT_PROXY:
        yt.config['proxy']['url'] = options.proxy

    q = Queue()

    avia_directions = []
    for departure_city_id in DEPARTURES:
        tds = AviaDirectionNational.objects.filter(
            departure_settlement_id=departure_city_id,
            national_version=NATIONAL_VERSION
        ).order_by(
            '-popularity'
        )[:25]

        for t in tds:
            avia_directions.append(t)

    for x, avia_direction in enumerate(avia_directions):
        # Если в базе нет реальных данных
        min_price_filter = {
            'national_version': NATIONAL_VERSION,
            'departure_settlement': avia_direction.departure_settlement,
            'arrival_settlement': avia_direction.arrival_settlement
        }

        min_price_exclude = {
            'eventtime': FAKE_EVENTTIME
        }

        if not MinPrice.objects.filter(**min_price_filter).exclude(**min_price_exclude).count():
            q.put((x, len(avia_directions), avia_direction, log, proxies))

    log.info('*** Fill %s directions' % len(avia_directions))
    for i in range(MAX_THREADS):
        worker = Thread(target=process_top_direction, args=(q, ))
        worker.setDaemon(True)
        worker.start()

    q.join()

    # Обновим дату последнего обновления
    setting.value = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    setting.save()

    log.info('*** Done')


class IncrementalResult(object):
    def __init__(self):
        self.airlines = {}
        self.airports = {}
        self.aircraft = {}
        self.flights = []
        self.outbound = []
        self.homebound = []

        self.vendors = {}
        self.fares = {}

        self.is_done = False

    def format_info(self):
        n_outbound = len(self.outbound)
        n_homebound = len(self.homebound)
        n_flights = len(self.flights)
        n_fares = 0

        for f in self.fares.itervalues():
            n_fares += len(f['f'])

        return '%i outbound, %i homebound, %i flights, %i fares' % (
            n_outbound, n_homebound, n_flights, n_fares
        )

    def as_int_map(self, m):
        assert(all(k.isdigit() for k in m.iterkeys()))
        return dict((int(k), v) for k, v in m.iteritems())

    def is_int_range_map(self, m):
        return all(k in m for k in xrange(len(m)))

    def extra_result(self, extra):
        if 'search' in extra:
            # airlines/airports/aircraft are just maps, update() is enough to track the result set
            self.airlines.update(extra['airlines'])
            self.airports.update(extra['airports'])
            self.aircraft.update(extra['aircraft'])

            # ...and flights/outbound/homebound are lists, extra results can be concatted
            self.flights.extend(extra['search']['flights'])
            self.outbound.extend(extra['search']['outbound'])
            self.homebound.extend(extra['search']['homebound'])

        if 'fares' in extra:
            self.vendors.update(extra['vendors'])
            # TBD: updating fare mapping becomes more sophisticated when /api/v1/fetch is finished
            self.fares.update(self.as_int_map(extra['fares']))
            assert(self.is_int_range_map(self.fares))

        if 'is_done' in extra:
            self.is_done = extra['is_done']


def main():
    create_current_file_run_log()
    _main()
