# -*- encoding: utf-8 -*-

import travel.avia.admin.init_project  # noqa

import logging
import requests
import urllib

from collections import defaultdict
from datetime import datetime, timedelta
from optparse import OptionParser

from django.conf import settings

from travel.avia.library.python.avia_data.models import AviaDirectionNational, MinPrice
from travel.avia.library.python.common.models.geo import CityMajority, Station, Settlement, StationCode, Country
from travel.avia.library.python.common.models.currency import Currency
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log


log = logging.getLogger(__name__)
create_current_file_run_log()

FAKE_EVENTTIME = datetime(2000, 1, 1)
LIVESTORE_ROOT = 'https://yandexapi.dohop.com'
PER_AIRPORT_TEMPLATE = '{api_url}/livestore/yandex,dohop/{language}/{user_country}/per-airport/{departure_airports}/{arrival_airports}/{date_from}/{date_to}?{qs}'
MAX_DESTINATIONS = 30
MAX_RETRY = 5
PASSENGERS_KEY = '1_0_0'
DEFAULT_NATIONAL_COUNTRIES = [
    ['kz', 'kz'],
    ['tr', 'tr'],
    ['ua', 'ua'],
    ['com', 'at,be,bg,hu,gb,gr,de,dk,it,ie,es,cy,lu,lv,lt,mt,nl,pt,pl,ro,si,sk,fr,fi,hr,cz,se,ee,is']
]


def get_cities_ids(majority, country_codes):
    stations = Station.objects.filter(
        t_type__code='plane',
        settlement_id__isnull=False,
        hidden=False,
        settlement__majority__lt=majority,
        country__code__in=country_codes,
    )

    return {s.settlement for s in stations}


def get_settlement_iatas(settlement):
    return [a.iata for a in settlement.get_all_iata_airports() if a.iata and a.t_type.code == 'plane']


def make_stay_string(date_forward, date_backward):
    stay = (date_backward - date_forward).days if date_backward else None

    if stay:
        stay_from = stay - 7
        # не может быть быть меньше нуля
        if stay_from < 0:
            stay_from = 0

        stay_to = stay + 7

        stay_string = '{stay_from}-{stay_to}'.format(
            stay_from=stay_from,
            stay_to=stay_to
        )
    else:
        stay_string = None

    return stay_string


def iata_to_point(iata):
    try:
        station_code = StationCode.objects.get(system__code='iata', code=iata)

    except StationCode.DoesNotExist:
        return None

    return station_code.station


def iata_to_point_key(iata):
    point = iata_to_point(iata)

    if point:
        # Постараемся отдать point_key города
        return point.settlement.point_key if point.settlement else point.point_key


def extract_prices_from_fares(fares, national_currency, national_version):
    min_prices = []
    try:
        currency = Currency.objects.get(code=national_currency)
    except Currency.DoesNotExist:
        log.warning('Currency "%s" not found for "%s"', national_currency, national_version)
        return min_prices

    for fare in fares:
        iata_from = fare['a']
        iata_to = fare['b']

        from_point_key = iata_to_point_key(iata_from)
        to_point_key = iata_to_point_key(iata_to)

        if not from_point_key or not to_point_key:
            continue

        n_legs_out = fare.get('n_legs_out', 0)
        n_legs_home = fare.get('n_legs_home', 0)

        price = int(float(fare['conv_fare']))

        date_forward = datetime.strptime(fare['d1'], "%Y-%m-%d")
        day_of_week = date_forward.weekday()

        min_price = MinPrice(
            departure_settlement=Settlement.get_by_key(from_point_key),
            arrival_settlement=Settlement.get_by_key(to_point_key),
            price=price,
            currency=currency,
            eventtime=FAKE_EVENTTIME,
            date_forward=date_forward,
            date_backward=datetime.strptime(fare.get('d2'), "%Y-%m-%d"),
            passengers='1_0_0',
            routes='',
            national_version=national_version,
            direct_flight=(n_legs_out + n_legs_home) == 0,
            day_of_week=day_of_week,
        )

        min_prices.append(min_price)

    return min_prices


def uniq_min_prices(bulk_min_prices):
    min_prices = defaultdict(list)

    for min_price in bulk_min_prices:
        min_price_key = '{key_from}_{key_to}_{date_forward}_{date_backward}_{passengers}_{direct_flight}'.format(
            key_from=min_price.departure_settlement,
            key_to=min_price.arrival_settlement,
            date_forward=min_price.date_forward.strftime('%Y-%m-%d'),
            date_backward=min_price.date_backward.strftime('%Y-%m-%d'),
            passengers=min_price.passengers,
            direct_flight=min_price.direct_flight,
        )
        min_prices[min_price_key].append(min_price)

    uniq_prices = []

    for key, min_price_list in min_prices.items():
        uniq_prices.append(min(min_price_list, key=lambda x: x.price))

    return uniq_prices


def save_min_price(bulk_min_prices, national_version):
    log.info('Clean existing data')
    MinPrice.objects.filter(national_version=national_version, eventtime=FAKE_EVENTTIME).delete()

    bulk_min_prices = uniq_min_prices(bulk_min_prices)
    log.info('Save %s prices', len(bulk_min_prices))
    MinPrice.objects.bulk_create(
        bulk_min_prices,
        batch_size=1000
    )


def fill_prices(national_version, raw_country_codes):
    national_currency = settings.AVIA_NATIONAL_CURRENCIES.get(national_version)

    if national_currency:
        log.info('*** national_currency: %s', national_currency)
    else:
        log.info('*** No default currency for %s national version', national_version)
        return

    country_codes = [c.code.strip() for c in Country.objects.filter(code__in=raw_country_codes.split(','))]

    if country_codes:
        log.info('*** Use country codes after check: %s', ', '.join(country_codes))

    else:
        log.info('*** Empty country codes after check: %s', raw_country_codes)
        return

    national_langs = settings.SITEMAP_NATIONAL_LANGS.get(national_version)

    if national_langs:
        log.info('*** Use national langs %s', ','.join(national_langs))
    else:
        log.info('*** No default langs for %s', ','.join(national_version))
        return

    settlements_from = []

    for country_code in country_codes:
        settlements = get_cities_ids(
            CityMajority.COMMON_CITY_ID,
            [country_code]
        )

        settlements_from = settlements_from + list(settlements)

    bulk_min_prices = []

    for settlement_from in settlements_from:
        iatas_from = get_settlement_iatas(settlement_from)
        top_directions = AviaDirectionNational.objects.filter(
            departure_settlement=settlement_from,
            national_version=national_version,
        ).order_by('-popularity')[:MAX_DESTINATIONS]

        for top_direction in top_directions:
            iatas_to = get_settlement_iatas(top_direction.arrival_settlement)

            min_price_count = MinPrice.objects.filter(
                departure_settlement=settlement_from,
                arrival_settlement=top_direction.arrival_settlement,
                national_version=national_version,
            ).exclude(
                eventtime=FAKE_EVENTTIME
            ).count()

            if min_price_count > 0:
                log.info('Skip from %s to %s', ','.join(iatas_from), ','.join(iatas_to))
                continue

            date_forward_from = datetime.today() + timedelta(days=1)
            date_forward_to = date_forward_from + timedelta(days=30)
            date_backward_from = datetime.today() + timedelta(days=7)

            query_string_data = {
                'currency': national_currency,
                'fare-format': 'full',
                'b_max': 8,
                'stay': make_stay_string(date_forward_from, date_backward_from)
            }

            log.info('Ask livestore from %s to %s', ','.join(iatas_from), ','.join(iatas_to))
            for lang in national_langs:
                url = PER_AIRPORT_TEMPLATE.format(
                    api_url='https://yandexapi.dohop.com/api/v2',
                    language=lang,
                    user_country=country_code,
                    departure_airports=','.join(iatas_from),
                    arrival_airports=','.join(iatas_to),
                    date_from=date_forward_from.strftime('%Y-%m-%d'),
                    date_to=date_forward_to.strftime('%Y-%m-%d'),
                    qs=urllib.urlencode(query_string_data)
                )

                for x in range(MAX_RETRY):
                    try:
                        r = requests.get(url, timeout=3)
                        json_data = r.json()
                    except:
                        log.info('Bad answer from live store. Retry %s/%s.', x + 1, MAX_RETRY)
                        continue

                    break

                for min_price in extract_prices_from_fares(json_data.get("fares", []), national_currency, national_version):
                    bulk_min_prices.append(min_price)

    save_min_price(bulk_min_prices, national_version)


def run_fill_prices(national_data):
    for national_version, raw_country_codes in national_data:
        fill_prices(national_version, raw_country_codes)


def main():
    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-c', '--country')
    optparser.add_option('-n', '--national')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('*** Start')

    if options.national and options.country:
        national_data = [
            [options.national, options.country]
        ]
    else:
        national_data = DEFAULT_NATIONAL_COUNTRIES

    run_fill_prices(national_data)

    log.info('*** Done')
