# -*- encoding: utf-8 -*-
import logging
import re
from optparse import OptionParser


from travel.avia.admin.lib.logs import add_stdout_handler
from travel.avia.admin.lib.yt_helpers import AviaYtClientFabric

log = logging.getLogger(__name__)

VALID_CLID_RE = re.compile(r'^[0-9\-]*$')
REPLACE_CLID = {
    '2255400-225routes/saint-petersburg--dubai/': '2255400-225',
}


def main():
    import travel.avia.admin.init_project  # noqa

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-s', '--source', dest='source', help='Redirect log source table')
    optparser.add_option('-d', '--destination', dest='destination', help='Destination table')
    optparser.add_option('-b', '--backup', dest='backup', help='Backup table')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    if not options.source:
        optparser.print_help()
        optparser.error('Source required')

    if not options.destination:
        optparser.print_help()
        optparser.error('Destination required')

    ytc = AviaYtClientFabric().create()

    if not ytc.exists(options.source):
        optparser.error('Unknown source: {}'.format(options.source))

    if ytc.exists(options.destination):
        optparser.error('Destination already exists: {}'.format(options.destination))

    log.info('Start')
    log.info('Source: %s', options.source)
    log.info('Destination: %s', options.destination)
    log.info('Backup: %s', options.backup)

    with ytc.Transaction():
        ytc.lock(options.source, waitable=True)

        log.info('From %s to %s', options.source, options.destination)

        ytc.create('table', options.destination, attributes={
            'schema': ytc.get_attribute(options.source, 'schema'),
            'processed': ytc.get_attribute(options.source, 'processed'),
            'compression_codec': ytc.get_attribute(options.source, 'compression_codec'),
            'optimize_for': ytc.get_attribute(options.source, 'optimize_for'),
        })

        if options.backup:
            ytc.copy(options.source, options.backup)
            log.info('Backup %s', options.backup)

        log.info('Processing %s (%d) ...', options.source, ytc.row_count(options.source))
        rows = []
        for row in ytc.read_table(options.source):
            if row['clid'] and row['clid'] != 'j:null':
                if row['clid'] in REPLACE_CLID:
                    log.info('Replace clid "%s" with "%s"', row['clid'], REPLACE_CLID[row['clid']])
                    row['clid'] = REPLACE_CLID[row['clid']]
                elif not VALID_CLID_RE.match(row['clid']):
                    log.info('Clear clid "%s"', row['clid'])
                    row['clid'] = None

            rows.append(row)

        log.info('Writing %s (%d) ...', options.destination, len(rows))
        ytc.write_table(options.destination, rows)

    log.info('Done')
