# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import travel.avia.admin.init_project  # noqa

import logging
import math
import os

from datetime import datetime, timedelta
from optparse import OptionParser
from unidecode import unidecode
from urllib import quote_plus
from subprocess import check_call

from django.conf import settings
from django.template import Template, Context
from django.utils.encoding import smart_str

from travel.avia.library.python.avia_data.models import TopDirection
from travel.avia.library.python.avia_data.models.prices import MinPrice
from travel.avia.library.python.avia_data.models.images import SettlementBigImage
from travel.avia.library.python.common.utils.geo import great_circle_distance_km

from travel.avia.admin.lib.logs import create_current_file_run_log, add_stdout_handler
from travel.avia.admin.lib.s3sync import s3sync
from utils.yandex_turbo import TurboFeed


log = logging.getLogger(__name__)
file_prefix = os.path.join(settings.MEDIA_ROOT, 'avia')

TITLE_TEMPLATE = Template('Дешёвые авиабилеты {{ fromPoint }} – {{ toPoint }} на Яндекс.Авиабилетах')
DESCRIPTION_TEMPLATE = Template('Авиабилеты {{ fromPoint }} – {{ toPoint }} по выгодным ценам. Поиск прямых рейсов и вариантов с пересадками, \
удобное сравнение предложений. Выбирайте и покупайте онлайн — билеты на самолет по маршруту {{ fromPoint }} – {{ toPoint }} на Яндекс.Авиабилетах.')

BODY_TEMPLATE = Template(" \
    <table> \
        {% if distance %} \
        <tr> \
            <td>Расстояние</td> \
            <td>{{ distance }} км</td> \
        </tr> \
        {% endif %} \
        {% if month_price %} \
        <tr> \
            <td>Цена на ближайшие 30 дней</td> \
            <td><a href=\"https://avia.yandex.ru/search/?fromId={{ fromId }}&toId={{ toId }}&when={{ month_price.date_forward|date:'Y-m-d' }}\">от {{ month_price.price }} руб.</a></td> \
        </tr> \
        {% endif %} \
        {% if year_price %} \
        <tr> \
            <td>Цена на ближайший год</td> \
            <td><a href=\"https://avia.yandex.ru/search/?fromId={{ fromId }}&toId={{ toId }}&when={{ year_price.date_forward|date:'Y-m-d' }}\">от {{ year_price.price }} руб.</a></td> \
        </tr> \
        {% endif %} \
    </table> \
")

RSS_CHANNEL_LIMIT = 500


class DirectionFeed(TurboFeed):
    title = 'Яндекс.Авиабилеты: поиск и покупка дешевых авиабилетов'
    description = 'Быстрый и удобный поиск билетов на самолет. Яндекс.Авиабилеты помогут найти нужные рейсы, сравнить цены, подобрать самые выгодные предложения и купить билеты на выбранный самолет онлайн.'  # noqa
    link = 'https://avia.yandex.ru/'
    logo = 'https://yastatic.net/avia-frontend/_/zgGqzO5DI7NaxJz-XhQOrJKXAuw.png'
    national_version = 'ru'
    country_id = settings.RUSSIA_GEO_ID
    item_pubdate = datetime.now()

    def __init__(self, national_version, country_id, directions, month_prices, year_prices, images):
        super(DirectionFeed, self).__init__()
        self.national_version = national_version
        self.country_id = country_id
        self.directions = directions
        self.month_prices = month_prices
        self.year_prices = year_prices
        self.images = images

    def item_turbo_logo(self, item):
        return self._get_image(item['direction'].arrival_settlement.id)

    def item_turbo_header_text(self, item):
        direction = item['direction']

        return TITLE_TEMPLATE.render(Context({
            'fromPoint': direction.departure_settlement.L_title(lang='ru'),
            'toPoint': direction.arrival_settlement.L_title(lang='ru'),
        }))

    def item_title(self, item):
        direction = item['direction']

        title = '{fromPoint} - {toPoint}'.format(
            fromPoint=direction.departure_settlement.L_title(lang='ru'),
            toPoint=direction.arrival_settlement.L_title(lang='ru'),
        )

        return super(DirectionFeed, self).item_title(title)

    def item_description(self, item):
        direction = item['direction']

        description = DESCRIPTION_TEMPLATE.render(Context({
            'fromPoint': direction.departure_settlement.L_title(lang='ru'),
            'toPoint': direction.arrival_settlement.L_title(lang='ru'),
        }))

        return super(DirectionFeed, self).item_description(description)

    def item_link(self, item):
        direction = item['direction']

        departure = direction.departure_settlement.iata.lower() \
                if direction.departure_settlement.iata else direction.departure_settlement_id

        arrival = direction.arrival_settlement.iata.lower() \
                if direction.arrival_settlement.iata else direction.arrival_settlement_id

        appendix = quote_plus(unidecode(
            '%s-%s' % (
                direction.departure_settlement.L_title(lang='ru').lower(),
                direction.arrival_settlement.L_title(lang='ru').lower(),
            )
        ).replace("'", ""))

        location_url = '{main_domain}/routes/{departure}/{arrival}/{appendix}/'.format(
            main_domain='https://avia.yandex.ru',
            departure=departure,
            arrival=arrival,
            appendix=appendix,
        )

        return location_url

    def item_turbo_body(self, item):
        direction = item['direction']

        return BODY_TEMPLATE.render(Context({
            'year_price': item['year_price'],
            'month_price': item['month_price'],
            'distance': item['distance'],
            'fromId': 'c{}'.format(direction.departure_settlement_id),
            'toId': 'c{}'.format(direction.arrival_settlement_id),
        }))

    def items(self):
        return [{
            'direction': d,
            'distance': self._get_distance(d),
            'month_price': self._find_direction_price(d, self.month_prices),
            'year_price': self._find_direction_price(d, self.year_prices),
        } for d in self.directions]

    def _find_direction_price(self, direction, prices_dict):
        key = direction.departure_settlement_id + direction.arrival_settlement_id
        return prices_dict[key] if key in prices_dict else None

    def _get_image(self, settlement_id):
        if settlement_id in self.images:
            return self.images[settlement_id]

        return None

    def _get_distance(self, direction):
        try:
            return int(round(great_circle_distance_km(direction.departure_settlement, direction.arrival_settlement)))
        except:
            return None


def _get_prices(national_version, delta):
    prices_dict = {}

    prices = MinPrice.objects.filter(
        national_version=national_version,
        date_forward__gte=datetime.now(),
        date_forward__lte=datetime.now() + delta,
    ).order_by(
        'price',
        'date_forward',
    )

    for mp in prices:
        key = mp.departure_settlement_id + mp.arrival_settlement_id
        if key not in prices_dict:
            prices_dict[key] = mp

    return prices_dict


def _get_directions(national_version, country_id):
    return TopDirection.objects.filter(
        national_version=national_version,
    ).select_related(
        'departure_settlement', 'arrival_settlement',
        'departure_settlement__new_L_title', 'arrival_settlement__new_L_title',
    ).order_by(
        '-count',
    )


def _get_images(settlement_ids):
    images = SettlementBigImage.objects.filter(settlement_id__in=settlement_ids)
    return {
        image.settlement_id: '{}/{}'.format(
            image.url2,
            'offer-desktop'
        ) for image in images
    }


def generate():
    # Делаем пробную версию ru/ru-RU
    # Если это даст какие-то результаты, то займемся другими нац. версиями и переводами
    national_version = 'ru'

    log.info('Start create feeds directiories')
    check_call('mkdir -p "%s"' % os.path.join(file_prefix, 'feeds', national_version), shell=True)
    log.info('Finish crete feeds directiories')

    directions = _get_directions(national_version, settings.RUSSIA_GEO_ID)
    month_prices = _get_prices(national_version, timedelta(days=30))
    year_prices = _get_prices(national_version, timedelta(days=365))
    images = _get_images(set([d.arrival_settlement_id for d in directions]))

    for channel_number in xrange(int(math.ceil(len(directions) * 1.0 / RSS_CHANNEL_LIMIT))):
        feed = DirectionFeed(
            directions=directions[channel_number * RSS_CHANNEL_LIMIT:(channel_number + 1) * RSS_CHANNEL_LIMIT],
            national_version=national_version,
            country_id=settings.RUSSIA_GEO_ID,
            month_prices=month_prices,
            year_prices=year_prices,
            images=images,
        )

        feed.add_analytics('Yandex', 13979182)
        feed.add_analytics('Yandex', 44999593)

        turbo = feed.get_feed(feed)

        save_to_file(
            os.path.join(file_prefix, 'feeds', national_version, 'rss_{}.xml'.format(channel_number)),
            smart_str(turbo.writeString('utf-8'))
        )

    s3sync('feeds')


def save_to_file(filepath, content):
    with open(filepath, 'wb') as f:
        f.write(content)
        f.flush()
        os.fsync(f)


def main():
    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)
    create_current_file_run_log()

    log.info('Start')
    generate()
    log.info('Finish')
