import travel.avia.admin.init_project  # noqa

import argparse
import logging
import os
from datetime import datetime, timedelta

from django.conf import settings

import yt.wrapper as yt
from library.python import resource

from travel.avia.admin.lib.logs import create_current_file_run_log, add_stdout_handler
from travel.avia.admin.lib.statface_helpers import post_data_to_stat
from travel.avia.admin.lib.yt_helpers import yt_client_fabric
from travel.avia.admin.lib.yql_helpers import get_yql_operation_url
from yql.api.v1.client import YqlClient


logger = logging.getLogger(__name__)
create_current_file_run_log()


ALLOWED_ENVS = ['production', 'dev']
QUERY_FILE = os.path.join('resfs/file', os.path.dirname(__file__), 'data', 'heater_stat.sql')
STAT_NAME = 'ticket.yandex.ru/Others/HeaterStat'
STAT_BETA_NAME = 'ticket.yandex/Others/HeaterStat'
YT_ROOT = '//home/logfeller/logs'


_DATE_FORMAT = '%Y-%m-%d'
_ONE_DAY = timedelta(days=1)
_YESTERDAY = datetime.today() - _ONE_DAY


def _parse_date_argument(value, default=_YESTERDAY):
    return datetime.strptime(value, _DATE_FORMAT) if value else default


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--left-date', default=None)  # Yesterday by default
    parser.add_argument('--right-date', default=None)  # Yesterday by default
    parser.add_argument('--skip-env-check', action='store_true')
    parser.add_argument('--verbose', action='store_true')

    return parser.parse_args()


def process_result(yt_client, table, operation):
    logger.info('Start process result')
    if not operation.is_ok:
        logger.error('Operation %s failed', operation.id)
        if operation.errors:
            for error in operation.error:
                logger.error(' - %s', str(error))
        return

    data = '[{}]'.format(
        ','.join(yt_client.read_table(table, format=yt.JsonFormat(), raw=True))
    )

    if data == '[]':
        logger.warning('Nothing to upload to to StatFace %s')
        return

    if settings.ENVIRONMENT == 'production':
        beta = False
        name = STAT_NAME
    else:
        beta = True
        name = STAT_BETA_NAME

    logger.info('Upload data to StatFace %s', data)
    post_data_to_stat(name, data, beta=beta, raw_data=True)
    logger.info('Upload table %s succeed', table)


def main():
    args = parse_args()
    if args.verbose:
        add_stdout_handler(logger)

    logger.info('Start')

    if not (args.skip_env_check or settings.ENVIRONMENT in ALLOWED_ENVS):
        logger.info('Can work only in %s', ', '.join(ALLOWED_ENVS))
        return

    query = resource.find(QUERY_FILE)

    yql_client = YqlClient(token=settings.YQL_TOKEN)
    yt_client = yt_client_fabric.create()

    left_date = _parse_date_argument(args.left_date, _YESTERDAY)
    right_date = _parse_date_argument(args.right_date, _YESTERDAY)

    current_date = left_date
    yql_requests = []
    while current_date <= right_date:
        current_date_str = current_date.strftime(_DATE_FORMAT)
        logger.info('Process %s', current_date_str)
        tariff_path = os.path.join(YT_ROOT, 'rasp-tariffs-log', '1d', current_date_str)
        avia_path = os.path.join(YT_ROOT, 'avia-users-search-log', '1d', current_date_str)
        rasp_path = os.path.join(YT_ROOT, 'rasp-users-search-log', '1d', current_date_str)
        wizard_path = os.path.join(YT_ROOT, 'avia-wizard-query-log', '1d', current_date_str)

        if not yt_client.exists(tariff_path):
            logger.info('Tariff log does not exist, skip')
            current_date += _ONE_DAY
            continue

        if not yt_client.exists(avia_path):
            logger.info('avia-users-search-log does not exist, skip')
            current_date += _ONE_DAY
            continue

        if not yt_client.exists(rasp_path):
            logger.info('rasp-users-search-log does not exist, skip')
            current_date += _ONE_DAY
            continue

        if not yt_client.exists(rasp_path):
            logger.info('rasp-users-search-log does not exist, skip')
            current_date += _ONE_DAY
            continue

        current_table = yt_client.create_temp_table()
        argument_string = '\n'.join([
            '$date="{}";'.format(current_date_str),
            '$price_path="{}";'.format(tariff_path),
            '$avia_searches_path="{}";'.format(avia_path),
            '$rasp_searches_path="{}";'.format(rasp_path),
            '$wizard_searches_path="{}";'.format(wizard_path),
            '$output_path="{}";'.format(current_table),
        ])
        current_query = '{arguments}\n{query}'.format(
            arguments=argument_string,
            query=query,
        )

        operation = yql_client.query(current_query, syntax_version=1)
        operation.run()
        logger.info('Output table: %s', current_table)
        logger.info('YQL operation: %s', get_yql_operation_url(operation))
        yql_requests.append((operation, current_table))

        current_date += _ONE_DAY

    for request, table in yql_requests:
        request.wait_progress()
        process_result(yt_client, table, request)

    logger.info('End')
