# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

from HTMLParser import HTMLParser
import logging
import requests
import sys
from datetime import datetime
from lxml import etree
from optparse import OptionParser

from django.db import transaction
from django.core.validators import URLValidator
from django.core.exceptions import ValidationError
from django.utils.html import strip_tags

from travel.avia.admin.lib.feeds import BadCompanyIATA, company_by_code
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.text import check_dirty_lang
from travel.avia.library.python.avia_data.models.review import FlightReview, FlightReviewSource
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.utils.safe_xml_parser import safe_xml_fromstring


try:
    from requests.packages.urllib3.exceptions import InsecureRequestWarning
    requests.packages.urllib3.disable_warnings(InsecureRequestWarning)
except ImportError:
    pass


log = logging.getLogger(__name__)
create_current_file_run_log()

AIRLINES_INFORM_URL_TEMPLATE = 'http://www.airlines-inform.ru/about_airline/rss.php?IATA=%s'
SOURCE, _SC = FlightReviewSource.objects.get_or_create(code='airlines-inform.ru')


def get_all_iatas():
    return Company.objects.filter(iata__isnull=False).values_list('iata', flat=True).distinct()


def get_feed(url):
    r = requests.get(url, verify=False)

    if r.status_code == 200:
        return r.content.decode('cp1251').encode('utf-8')
    else:
        log.error('Can not get feed for [%s], status code: %d', url, r.status_code)


def store_review(review, airline_code):
    try:
        airline = company_by_code(airline_code, log)

    except BadCompanyIATA:
        return

    author_name = strip_tags(review.findtext('title'))

    review_datetime = datetime.strptime(review.findtext('pubDate'), '%Y-%m-%d %H:%M:%S')
    review_content = HTMLParser().unescape(
        strip_tags(
            review.findtext('description')
        )
    )

    try:
        dirty_lang_class = check_dirty_lang(review_content)

    except Exception:
        log.exception("WARNING")
        return

    except:
        log.error('Error while dirty lang check')
        return

    if dirty_lang_class:
        log.info('%s: %s' % (dirty_lang_class, review_content))

    review_url = review.findtext('link').strip()
    try:
        URLValidator(review_url)

    except ValidationError:
        log.error('Bad url %s', review_url)
        return

    review_id = int(review_url.split('#message')[-1])

    FlightReview.objects.get_or_create(
        review_id=review_id,
        source=SOURCE,
        defaults={
            'review_content': review_content,
            'review_datetime': review_datetime,
            'review_url': review_url,
            'author_name': author_name,
            'airline': airline,
            'dirty_lang_class': dirty_lang_class
        }
    )


@transaction.atomic
def store_feed(xml, airline_code):
    # Kill encoding from xml
    xml = xml.replace('encoding="Windows-1251"', '').strip()

    try:
        tree = safe_xml_fromstring(xml)
    except etree.XMLSyntaxError:
        log.info('Bad XML syntax')
        return

    for review in tree.xpath('//channel/item'):
        try:
            store_review(review, airline_code)

        except Exception:
            log.exception('ERROR')
            continue


def main():
    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('*** Start')

    if not SOURCE.enabled:
        log.info('%s disabled' % SOURCE)
        sys.exit(0)

    all_iatas = get_all_iatas()

    for x, iata in enumerate(all_iatas):
        airlines_inform_url = AIRLINES_INFORM_URL_TEMPLATE % iata
        log.info('%s/%s Get feed for %s: %s', x, len(all_iatas), iata, airlines_inform_url)
        feed_content = get_feed(airlines_inform_url)

        if feed_content == 'error: wrong IATA':
            log.info('Empty answer for %s', iata)
            continue

        if feed_content:
            store_feed(feed_content, iata)
        else:
            log.info('Failed to get feed for %s', iata)

    log.info('*** Done')
