# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

from HTMLParser import HTMLParser
import logging
import requests
import sys
from datetime import datetime
from optparse import OptionParser

from django.db import transaction
from django.core.validators import URLValidator
from django.core.exceptions import ValidationError
from django.utils.html import strip_tags

from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.feeds import BadCompanyIATA, company_by_code
from travel.avia.admin.lib.text import check_dirty_lang
from travel.avia.library.python.avia_data.models.review import FlightReview, FlightReviewSource
from travel.avia.library.python.common.utils.safe_xml_parser import safe_xml_fromstring

log = logging.getLogger(__name__)
create_current_file_run_log()

AVIAKOMPANIYA_URL = 'http://aviakompaniya.info/airlines_comments100.xml'
SOURCE, _SC = FlightReviewSource.objects.get_or_create(code='aviakompaniya.info')


def get_feed(url):
    r = requests.get(url)

    if r.status_code == 200:
        return r.content


def store_review(review):
    airline_code = review.findtext('IATA') or review.findtext('ICAO')

    if not airline_code:
        return

    try:
        airline = company_by_code(airline_code, log)

    except BadCompanyIATA:
        return

    author_name = strip_tags(review.findtext('author'))

    review_datetime = datetime.strptime(review.findtext('date'), '%d.%m.%Y')
    review_content = HTMLParser().unescape(
        strip_tags(
            review.findtext('comment').replace('<br />', '\n').replace('</p>', '\n')
        )
    )

    try:
        dirty_lang_class = check_dirty_lang(review_content)

    except Exception:
        log.exception("ERROR")
        return

    except:
        log.error('Error while dirty lang check')
        return

    if dirty_lang_class:
        log.info('%s: %s' % (dirty_lang_class, review_content))

    review_url = review.findtext('comment_link').strip()
    try:
        URLValidator(review_url)

    except ValidationError:
        log.error('Bad url %s', review_url)
        return

    review_id = int(review.findtext('ID'))

    FlightReview.objects.get_or_create(
        review_id=review_id,
        source=SOURCE,
        defaults={
            'review_content': review_content,
            'review_datetime': review_datetime,
            'review_url': review_url,
            'author_name': author_name,
            'airline': airline,
            'dirty_lang_class': dirty_lang_class
        }
    )


@transaction.atomic
def store_feed(xml):
    # Kill encoding from xml
    xml = xml.replace('encoding="windows-1251"', '').strip()

    tree = safe_xml_fromstring(xml)
    for review in tree.xpath('//nodes/content'):
        try:
            store_review(review)

        except Exception:
            log.exception('ERROR')
            continue


def main():
    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-s', '--skip_source_enabled_check', action='store_true')
    optparser.add_option('-f', '--file')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('*** Start')

    if not SOURCE.enabled and not options.skip_source_enabled_check:
        log.info('%s disabled' % SOURCE)
        sys.exit(0)

    if options.file:
        with open(options.file) as f:
            feed_content = f.read()

    else:
        feed_content = get_feed(AVIAKOMPANIYA_URL)

    store_feed(feed_content)

    log.info('*** Done')
