# -*- encoding: utf-8 -*-
from functools import partial

from collections import defaultdict, namedtuple
from datetime import datetime, timedelta, date
from optparse import OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

BALANCE_BY_DAY_ROOT = '//home/avia/logs/avia-redir-balance-by-day-log'
NEW_SCHEME_START = date(2018, 3, 1)  # 1 марта перешли на новую схему. Раньше этой даты в логах может не быть нужных полей и агрегацию делать не нужно
PreparedKey = namedtuple('PreparedKey', ('event_date', 'billing_client_id', 'national_version', 'pp'))


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.conf import settings
    from django.db import transaction

    from travel.avia.library.python.avia_data.models import BalancePriceListRedirectPrepared, BalanceRedirectPrepeared
    from travel.avia.library.python.common.models.partner import Partner
    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper, safe_tables_for_daterange

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def filter_log(left_date, datetime_field, record):
        iso_eventtime = datetime.strptime(record.get(datetime_field), "%Y-%m-%d %H:%M:%S").date()
        if iso_eventtime >= left_date:
            yield record

    @transaction.atomic
    def write_balance(source_table, left_date):
        log.info('Save results')

        log.info('Cache partners')
        cached_partners = dict(
            Partner.objects.filter(billing_order_id__gt=0)
                .values_list('billing_order_id', 'id')
        )

        log.info('Delete records greater or equal: %s', left_date)
        BalanceRedirectPrepeared.objects.filter(eventdate__gte=left_date).delete()

        error_count = 0
        log.info('Read data from %s', source_table)

        prepared_data = defaultdict(int)

        for record in yt.read_table(source_table, format=yt.JsonFormat(), raw=False):
            try:
                datasource_id = record.get('DATASOURCE_ID')
                billing_order_id = record.get('BILLING_ORDER_ID')

                # Если нет BILLING_ORDER_ID, то делаем его из DATASOURCE_ID
                if not billing_order_id and datasource_id:
                    billing_order_id = int(datasource_id) + 1

                if not billing_order_id:
                    log.info('BILLING_ORDER_ID or DATASOURCE_ID not found')
                    continue

                iso_eventtime = datetime.strptime(record.get('ISO_EVENTTIME'), "%Y-%m-%d %H:%M:%S")

                partner_id = cached_partners.get(int(billing_order_id))

                if not partner_id:
                    log.info('Unknown billing_order_id: %r', billing_order_id)

                billing_order_id = int(billing_order_id)
                national_version = record.get('NATIONAL')
                pp = int(record.get('PP'))

                # RASPTICKETS-7681: При импорте данных из баланса интерпретировать 601 pp
                # как десктоп и сохранять его в базу как десктопный pp
                if pp == 601:
                    pp = 502

                filtered = bool(int(record.get('FILTER')))

                if not filtered:
                    prepared_key = (iso_eventtime.date(), billing_order_id, national_version, pp)
                    prepared_data[prepared_key] += 1

            except Exception as e:
                log.error("Error while import %s. source: %s", e.message, record)
                error_count += 1
                continue

        log.info("Data loaded with %s errors", error_count)

        log.info('Creating prepeared data')
        prepared_objects = []
        for prepared_key, count in prepared_data.items():
            eventdate, billing_order_id, national_version, pp = prepared_key
            prepared_object = BalanceRedirectPrepeared(
                eventdate=eventdate,
                billing_order_id=billing_order_id,
                national_version=national_version,
                pp=pp,
                count=count
            )

            prepared_objects.append(prepared_object)

        log.info('Save %s prepeared records', len(prepared_objects))
        BalanceRedirectPrepeared.objects.bulk_create(prepared_objects, batch_size=1000)

    @transaction.atomic
    def write_balance_new_scheme(source_table, booking_logs_table, left_date):
        left_date = max(NEW_SCHEME_START, left_date)
        log.info('Save new_scheme results')

        log.info('Cache partners')
        cached_partners = dict(
            Partner.objects.filter(billing_client_id__gt=0)
            .values_list('billing_client_id', 'id')
        )

        log.info('Delete records greater or equal: %s', left_date)
        BalancePriceListRedirectPrepared.objects.filter(event_date__gte=left_date).delete()

        error_count = 0
        log.info('Read data from %s', source_table)

        aggregated_clics = defaultdict(list)
        aggregated_payments = defaultdict(int)

        log.info('Collecting markers info')
        paid_markers = set(order['marker'] for order in yt.read_table(booking_logs_table) if order['status'] == 'paid')

        for record in yt.read_table(source_table, format=yt.JsonFormat(), raw=False):
            try:
                billing_client_id = int(record.get('BILLING_CLIENT_ID'))
                price = float(record.get('PRICE'))
                iso_event_time = datetime.strptime(record.get('ISO_EVENTTIME'), '%Y-%m-%d %H:%M:%S')

                partner_id = cached_partners.get(billing_client_id)

                if not partner_id:
                    log.warning('Unknown billing_client_id: %r', billing_client_id)

                national_version = record.get('NATIONAL')
                pp = int(record.get('PP'))
                marker = record.get('MARKER')

                # RASPTICKETS-7681: При импорте данных из баланса интерпретировать 601 pp
                # как десктоп и сохранять его в базу как десктопный pp
                if pp == 601:
                    pp = 502

                filtered = bool(int(record.get('FILTER')))

                if not filtered:
                    prepared_key = PreparedKey(iso_event_time.date(), billing_client_id, national_version, pp)
                    aggregated_clics[prepared_key].append(price)
                    if marker and marker in paid_markers:
                        aggregated_payments[prepared_key] += 1

            except Exception as e:
                log.error('Error while import %s. source: %s', e.message, record)
                error_count += 1
                continue

        log.info('Data loaded with %s errors', error_count)

        log.info('Creating prepeared data')
        prepared_objects = []
        for prepared_key, prices in aggregated_clics.iteritems():
            prepared_object = BalancePriceListRedirectPrepared(
                event_date=prepared_key.event_date,
                billing_client_id=prepared_key.billing_client_id,
                national_version=prepared_key.national_version,
                pp=prepared_key.pp,
                count=len(prices),
                cost=sum(prices),
                payments_count=aggregated_payments[prepared_key],
            )

            prepared_objects.append(prepared_object)

        log.info('Save %s prepeared records', len(prepared_objects))
        BalancePriceListRedirectPrepared.objects.bulk_create(prepared_objects, batch_size=1000)

    # BEGIN main()
    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-d', '--days', dest='days', type='int', help='number of days to import', default=1)

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)

        tmp_table = yt.create_temp_table()
        date_now = datetime.now().date()
        left_date = date_now - timedelta(days=options.days)

        # бывает, что маркеры приходят на 1-2 дня раньше
        booking_logs_left_date = date_now - timedelta(days=options.days + 2)

        log.info('Left date: %s', left_date.strftime('%Y-%m-%d'))

        balance_logs = safe_tables_for_daterange(
            yt, BALANCE_BY_DAY_ROOT, left_date, date_now
        )
        booking_logs = safe_tables_for_daterange(
            yt, settings.YT_PARTNER_BOOKING_ROOT, booking_logs_left_date, date_now
        )

        all_booking_logs_table = yt.create_temp_table()
        yt.run_map(
            partial(filter_log, left_date, 'created_at'),
            source_table=booking_logs,
            destination_table=all_booking_logs_table
        )

        if not balance_logs:
            log.info('No balance logs for daterange [%s; %s]', left_date, date_now)
            return

        yt.run_map(
            partial(filter_log, left_date, 'ISO_EVENTTIME'),
            source_table=balance_logs,
            destination_table=tmp_table,
        )

        write_balance(tmp_table, left_date)
        write_balance_new_scheme(tmp_table, all_booking_logs_table, left_date)

        yt.remove(tmp_table)
        yt.remove(all_booking_logs_table)

    except Exception:
        log.exception('Error:')
        return

    log.info('Done')
