# -*- encoding: utf-8 -*-
import json
import logging
import os
import sys
from collections import defaultdict, OrderedDict
from datetime import datetime, timedelta, date
from functools import partial
from itertools import chain
from threading import Thread
from Queue import Queue

from cachetools.func import lru_cache
import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

log = logging.getLogger(__name__)

BALANCE_BY_DAY_ROOT = '//home/avia/logs/avia-redir-balance-by-day-log'
BOOKING_LOG_ROOT = '//home/avia/logs/avia-partner-booking-log'
TABLE_TTL = timedelta(minutes=30)  # todo: задать 30 минут
NEW_SCHEME_START = date(2018, 3, 1)  # 1 марта перешли на новую схему. Раньше этой даты в логах может не быть нужных полей и агрегацию делать не нужно


def get_base_path(environment):
    if environment == 'production':
        return '//home/avia/logs/partnerka-reports'

    return '//home/avia/{}/partnerka-reports'.format(environment)


def format_destination_path(base, billing_client_id, start_date, end_date):
    return '%s/%s/%s_%s' % (
        base, billing_client_id, start_date, end_date
    )


def _parse_date(dt):
    return datetime.strptime(dt, '%Y-%m-%d').date()


def _parse_maybe_date(dt):
    return _parse_date(dt) if dt else None


class OrderedSet(OrderedDict):
    """
    В итоге чтобы получить значения, нужно будет преобразовать в список
    """
    def add(self, value):
        if value not in self:
            self[value] = None


def _stations_iatas_by_station_id():
    from travel.avia.library.python.common.models.geo import StationCode
    return dict(StationCode.objects.filter(system__code='iata')
                .values_list('station_id', 'code'))


def _station_id_by_iata():
    return {iata: sid for sid, iata in _stations_iatas_by_station_id().iteritems()}


def _stations_sirenas_by_station_id():
    from travel.avia.library.python.common.models.geo import StationCode
    return dict(StationCode.objects.filter(system__code='sirena')
                .values_list('station_id', 'code'))


def _station_id_by_sirena():
    return {sirena: sid for sid, sirena in _stations_sirenas_by_station_id().iteritems()}


def _airport_by_id():
    from travel.avia.library.python.common.models.geo import Station
    from travel.avia.library.python.common.models.transport import TransportType
    from django.db.models import Q

    station_ids = set(chain(
        _station_id_by_iata().itervalues(),
        _station_id_by_sirena().itervalues()
    ))
    iata_by_station_id = _stations_iatas_by_station_id()

    # hidden=False  # Разрешаем пока все, даже hidden

    qs = Q(id__in=station_ids) | Q(t_type_id=TransportType.PLANE_ID)

    return {
        s['id']: dict(s, iata=iata_by_station_id.get(s['id']))
        for s in Station.objects.filter(qs)
        .values('id', 'settlement_id', 'sirena_id')
    }


def _station2settlement():
    from travel.avia.library.python.common.models.geo import Station2Settlement
    return dict(Station2Settlement.objects.filter(station_id__in=_airport_by_id().keys()).values_list('station_id', 'settlement_id'))


def _airport_ids_by_settlement_id():
    by_settlement = defaultdict(OrderedSet)

    for airport in _airport_by_id().values():
        if airport['settlement_id']:
            by_settlement[airport['settlement_id']].add(airport['id'])

    for station_id, settlement_id in _station2settlement().iteritems():
        by_settlement[settlement_id].add(station_id)

    return {
        settlement_id: tuple(stations_ids)
        for settlement_id, stations_ids in by_settlement.iteritems()
    }


@lru_cache(maxsize=None)
def get_price_list():
    from travel.avia.library.python.common.cache.price_list import price_list_cache

    price_list_cache.pre_cache()
    return price_list_cache


@lru_cache(maxsize=None)
def get_point_reference():
    from travel.avia.library.python.common.models.geo import Settlement
    reference = {}
    settlements_by_id = {
        s['id']: s
        for s in Settlement.objects.all()
        .values('id', 'title_ru', 'title_en', 'iata', 'sirena_id')
    }
    airports_by_id = _airport_by_id()
    airport_ids_by_settlement_id = _airport_ids_by_settlement_id()

    def station_code(station):
        if not station['settlement_id']:
            return {}
        settlement = settlements_by_id.get(station['settlement_id'], {})
        if not settlement:
            return {}

        if station['iata']:
            return {'IATA': station['iata']}
        elif station['sirena_id']:
            return {'Sirena': station['sirena_id']}
        return {}

    for s in settlements_by_id.itervalues():
        settlement_ref = {
            'city': s['title_en'] or s['title_ru'],
            'IATA': '',
            'Sirena': '',
        }
        if s['iata']:
            settlement_ref['IATA'] = s['iata']
        else:
            airports = airport_ids_by_settlement_id.get(s['id'], [])
            if len(airports) == 1:
                station = airports_by_id.get(airports[0])
                if station:
                    settlement_ref.update(station_code(station))
            elif s['sirena_id']:
                settlement_ref['Sirena'] = s['sirena_id']
        reference[str(s['id'])] = settlement_ref

    for station_id, station in airports_by_id.iteritems():
        # половину марта мы не писли settlement_from_id и settlement_to_id
        # и города перелета определяем по point-ам из запроса, а там есть и станции
        if not station['settlement_id']:
            continue
        settlement = settlements_by_id.get(station['settlement_id'], {})
        if not settlement:
            continue
        settlement_ref = {
            'city': settlement['title_en'] or settlement['title_ru'],
            'IATA': '',
            'Sirena': '',
        }
        settlement_ref.update(station_code(station))
        reference[str(station_id)] = settlement_ref
    return reference


class ReportMaker(object):
    APPEND_ROUTE_QUERY = '''
        PRAGMA SimpleColumns;
        PRAGMA yt.InferSchema = '1';

        DECLARE $LeftDate AS String;
        DECLARE $RightDate AS String;
        DECLARE $TargetTable AS String;

        USE hahn;

        $GetRoute = ($routeStr) -> {
            RETURN String::JoinFromList(
                ListMap(
                    String::SplitToList($routeStr, ","),
                    ($x) -> {
                        RETURN String::SplitToList($x, ".")[0];
                    }
                ),
                "_"
            );
        };

        $GetRoutes = ($forward, $backward) -> {
            $forwardString = $GetRoute($forward);
            RETURN IF(
                $backward == "" OR $backward IS NULL,
                $forwardString,
                $forwardString || ";" || $GetRoute($backward)
            )
        };

        $Routes = (
            SELECT
            variant_id,
            $GetRoutes(SOME(forward), SOME(backward)) AS route
            FROM (
                %s
            ) AS show
            LEFT SEMI JOIN $TargetTable AS target
            ON show.variantId = target.variant_id
            GROUP BY show.variantId AS variant_id
        );

        INSERT INTO
            $TargetTable
        WITH TRUNCATE

        SELECT
            t.* WITHOUT variant_id, _other
        FROM (
            SELECT
                routes.route AS flight,
                report.*
            FROM $TargetTable AS report
            LEFT JOIN $Routes AS routes
            USING (variant_id)
        ) AS t;
    '''

    BEFORE_TODAY_ACTION_SHOW_LOG = '''
        SELECT
            *
        FROM
            RANGE(
                `logs/avia-action-show-log/1d`,
                $LeftDate,
                $RightDate
            )
    '''

    TODAY_ACTION_SHOW_LOG = '''
        SELECT
            *
        FROM
            RANGE(
                `logs/avia-action-show-log/30min`,
                CAST(CurrentUtcDate() as String) || 'T00:00:00',
                $RightDate || 'T23:59:59'
            )
    '''

    def __init__(self, yt_client_fabric, yql_client_fabric, price_list, point_reference, billing_client_to_order_id, platform_code_to_name, base_path):
        self._yt_client_fabric = yt_client_fabric
        self._yql_client_fabric = yql_client_fabric
        self._price_list = price_list
        self._point_reference = point_reference
        self._billing_client_to_order_id = billing_client_to_order_id
        self._platform_code_to_name = platform_code_to_name
        self._base_path = base_path

        self._yt_client = None
        self._yql_client = None

    def _get_yql_client(self):
        if self._yql_client is None:
            self._yql_client = self._yql_client_fabric.create()

        return self._yql_client

    def _get_yt_client(self):
        if self._yt_client is None:
            self._yt_client = self._yt_client_fabric.create()

        return self._yt_client

    def get_orders(self, left_date, right_date, billing_order_id):
        ytc = self._get_yt_client()
        return set(
            record['marker']
            for record in ytc.read_tables(
                ytc.tables_for_daterange(
                    BOOKING_LOG_ROOT, left_date, right_date + timedelta(days=1)
                ),
                columns=['billing_order_id', 'status', 'marker'],
            )
            if record['billing_order_id'] == billing_order_id and record['status'] == 'paid'
        )

    def append_routes(self, left_date, right_date, record_table, transaction_id):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        today = datetime.today().date()

        yql_parts = []
        if left_date < today:
            yql_parts.append(self.BEFORE_TODAY_ACTION_SHOW_LOG)
        if right_date >= today:
            yql_parts.append(self.TODAY_ACTION_SHOW_LOG)

        yql_client = self._get_yql_client()
        query = yql_client.query(
            self.APPEND_ROUTE_QUERY % 'UNION ALL'.join(yql_parts),
            syntax_version=1,
            title='[YQL] Append routes to partner report',
        )

        result = query.run(
            parameters=ValueBuilder.build_json_map({
                '$LeftDate': ValueBuilder.make_string(left_date.strftime('%Y-%m-%d')),
                '$RightDate': ValueBuilder.make_string(right_date.strftime('%Y-%m-%d')),
                '$TargetTable': ValueBuilder.make_string(record_table),
            }),
            transaction_id=transaction_id,
        )

        log.info('Append routes: %s', result.share_url)
        result.wait_progress()
        if not result.is_success:
            from travel.avia.admin.lib.yql_helpers import log_errors
            log_errors(result, log)
            raise ValueError('Operation failed')

    def fill_point_reference(self, record, row, direction):
        empty_point_reference = {'IATA': '', 'Sirena': '', 'city': ''}
        settlement_id = record.get('settlement_{}_id'.format(direction))

        if not settlement_id:
            # Получаем город из запроса. Нужно только до тех пор, пока не станем писать id городов тарификации
            point_key = record.get('{}ID'.format(direction.upper()))
            settlement_id = point_key[1:]  # тут на самом деле могут быть и станции, но они тоже есть в point_reference

        point = self._point_reference.get(settlement_id, empty_point_reference)
        row.update({
            '{}_IATA'.format(direction): point['IATA'],
            '{}_Sirena'.format(direction): point['Sirena'],
            '{}_city'.format(direction): point['city'],
        })

    def filter_log(self, left_date, billing_client_id, orders, extended_partner_report, record):
        if billing_client_id == record.get('BILLING_CLIENT_ID') and not record.get('FILTER'):
            iso_eventtime = datetime.strptime(record.get('ISO_EVENTTIME'), "%Y-%m-%d %H:%M:%S").date()
            if iso_eventtime >= left_date:
                _date, time = record.get('ISO_EVENTTIME').split()
                row = {
                    'date': _date,
                    'time': time,
                    'month': int(record['WHEN'][5:7]),  # 2018-02-24
                    'OW/RT': 'RT' if bool(record['RETURN_DATE']) else 'OW',
                    'adult_seats': record['ADULT_SEATS'],
                    'children_seats': record['CHILDREN_SEATS'],
                    'infant_seats': record['INFANT_SEATS'],
                    'price': record['PRICE'],
                    'marker': record['MARKER'],
                    'national_version': record['NATIONAL'],
                    'from_IATA': '',  # код города вылета, использованный при тарификации перелета.
                    'to_IATA': '',  # код города прилета, использованный при тарификации перелета.
                    'from_Sirena': '',  # код города вылета, использованный при тарификации перелета, если у него нет IATA-кода
                    'to_Sirena': '',  # код города прилета, использованный при тарификации перелета, если у него нет IATA-кода
                    'from_city': '',  # название города вылета на английском, использованный при тарификации перелета. Фоллбек на название города на русском.
                    'to_city': '',  # название города прилета на английском, использованный при тарификации перелета. Фоллбек на название города на русском.
                }

                self.fill_point_reference(record, row, direction='from')
                self.fill_point_reference(record, row, direction='to')

                if extended_partner_report:
                    row.update({
                        'sale': 'true' if record['MARKER'] in orders else 'false',
                        'offer_price': record['OFFER_PRICE'],
                        'variant_id': record['VARIANTID'],
                        'default_price':  float(self._price_list.get_click_price(
                            record['_REST']['settlement_from_id'], record['_REST']['settlement_to_id'],
                            record['NATIONAL'],
                            _parse_date(record['WHEN']), _parse_maybe_date(record['RETURN_DATE']),
                            record['ADULT_SEATS'], record['CHILDREN_SEATS'],
                            use_new_pricing=False,
                        )['price_cpa']) / 100,
                        'platform': self._platform_code_to_name.get(record['PP']),
                        'source': get_redirect_source(record),
                    })

                yield row

    def make_report(self, start_date, end_date, billing_client_id, extended_partner_report=False):
        import pytz

        from travel.avia.admin.lib.yt_helpers import table_is_empty

        left_date = max(NEW_SCHEME_START, start_date)
        log.info('Start for daterange [%s; %s]. Partner %s', left_date, end_date, billing_client_id)

        ytc = self._get_yt_client()
        try:
            destination_table = format_destination_path(self._base_path, billing_client_id, start_date, end_date)
            if ytc.exists(destination_table):
                log.info('Table "%s" already exist', destination_table)
                return
            else:
                log.info('Destination table: %s', destination_table)
                if extended_partner_report:
                    # YQL drops table TTL, so we create a temporary table
                    temp_table = ytc.create_temp_table()
                else:
                    temp_table = destination_table

                with ytc.Transaction(timeout=60000 * 5) as transaction:
                    expiration_time = pytz.UTC.localize(datetime.utcnow() + TABLE_TTL).isoformat()
                    ytc.create(
                        'table', path=destination_table, recursive=True,
                        attributes={'expiration_time': expiration_time}
                    )
                    balance_logs = ytc.tables_for_daterange(
                        BALANCE_BY_DAY_ROOT, left_date, end_date
                    )

                    if not balance_logs:
                        log.warning('No balance logs for date range [%s; %s]', left_date, end_date)
                        return

                    orders = self.get_orders(
                        left_date, end_date, self._billing_client_to_order_id.get(billing_client_id)
                    )
                    log.info('Got orders')

                    ytc.run_map(
                        partial(self.filter_log, left_date, billing_client_id, orders, extended_partner_report),
                        source_table=balance_logs,
                        destination_table=temp_table,
                    )
                    log.info('Got base report')

                    if extended_partner_report:
                        if table_is_empty(ytc, temp_table):
                            log.info('Intermediate table for extended report is empty. Skip adding routes')
                        else:
                            self.append_routes(
                                left_date, end_date, temp_table, transaction.transaction_id,
                            )

                            ytc.run_merge(temp_table, destination_table)
                            log.info('Routes have been appended')
                        ytc.remove(temp_table)

        except yt.YtHttpResponseError as exc:
            if str(exc).startswith('Cannot take lock'):
                log.info('Concurrent transaction is running')
            else:
                log.exception('Error:')
                raise
        except Exception:
            log.exception('Error:')
            raise
        log.info('Done')


# Костыль, чтобы не тащить Джангу на Ыть
def get_redirect_source(record):
    if record['UTM_SOURCE'] == 'rasp' and record['UTM_MEDIUM'] == 'redirect':
        return u'Яндекс.Расписания'

    if record['UTM_SOURCE'] == 'sovetnik' and record['UTM_CONTENT'] == 'redirect':
        return u'Советник Яндекс.Маркета'

    if record['UTM_SOURCE'] == 'wizard_ru' or record['UTM_SOURCE'] == 'unisearch_ru':
        if record['WIZARDREDIRKEY']:
            return u'Яндекс.Поиск'

        return u'Яндекс.Поиск через Яндекс.Авиа'

    return u'Яндекс.Авиабилеты'


@lru_cache(maxsize=None)
def get_billing_client_to_order_id():
    from travel.avia.library.python.common.models.partner import Partner
    return dict(
        Partner.objects.all().values_list('billing_client_id', 'billing_order_id')
    )


def _task_file():
    from django.conf import settings
    return os.path.join(settings.LOG_PATH, 'special', 'partnerka_reports.log')


def add_task(start_date, end_date, billing_client_id):
    task = json.dumps({
        'start_date': start_date,
        'end_date': end_date,
        'billing_client_id': billing_client_id,
    })
    with open(_task_file(), 'a') as f:
        f.write(task)
        f.write('\n')


def _unbuffered_stdin():
    for line in iter(sys.stdin.readline, None):
        if not line:
            break
        yield line


def create_report_maker():
    from django.conf import settings

    from travel.avia.admin.lib.yt_helpers import AviaYtClientFabric
    from travel.avia.admin.lib.yql_helpers import YqlClientFabric
    from travel.avia.admin.lib.pp import PLATFORM_CODE_TO_NAME

    return ReportMaker(
        AviaYtClientFabric(),
        YqlClientFabric(),
        get_price_list(),
        get_point_reference(),
        get_billing_client_to_order_id(),
        PLATFORM_CODE_TO_NAME,
        get_base_path(settings.ENVIRONMENT),
    )


class Worker(Thread):
    def __init__(self, report_maker, queue, logger):
        Thread.__init__(self)
        self._report_maker = report_maker
        self._queue = queue
        self._logger = logger

    def run(self):
        while True:
            args = self._queue.get()
            try:
                self._report_maker.make_report(*args)
            except KeyboardInterrupt:
                break
            except Exception:
                self._logger.exception('ERROR')

            finally:
                self._queue.task_done()


def start_polling(worker_count=5):
    from django.conf import settings  # noqa

    from travel.avia.admin.lib.feature_flags import extended_report_flag_by_partner_code
    from travel.avia.library.python.common.models.partner import Partner

    q = Queue()
    workers = []
    for _ in range(worker_count):
        worker = Worker(create_report_maker(), q, log)
        worker.setDaemon(True)
        worker.start()
        workers.append(worker)

    data = (json.loads(line) for line in _unbuffered_stdin())
    for idx, task in enumerate(data, start=1):
        log.info('Add %d task %r', idx, task)
        partner = Partner.objects.get(billing_client_id=task['billing_client_id'])
        q.put((
            _parse_date(task['start_date']),
            _parse_date(task['end_date']),
            task['billing_client_id'],
            extended_report_flag_by_partner_code(partner.code),
        ))

    q.join()
    log.info('Done')


def main():
    import travel.avia.admin.init_project  # noqa

    from travel.avia.admin.avia_scripts.utils.argument_parser import verbose_argument_parser
    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log

    args = verbose_argument_parser.parse_args()
    create_current_file_run_log()
    if args.verbose:
        add_stdout_handler(log)
    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    start_polling()
