# -*- encoding: utf-8 -*-
import sys

from collections import defaultdict
from datetime import datetime, timedelta
from optparse import OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

SHOW_LOG_PATH = '//home/logfeller/logs/avia-show-log'


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.db import transaction

    from travel.avia.library.python.avia_data.models import ShowLog
    from travel.avia.library.python.common.models.partner import Partner
    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import safe_tables_for_daterange, configure_wrapper

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    @transaction.atomic
    def write_showlog(show_log_items, left_date):
        log.info('Save results')

        log.info('Cache partners')
        cached_partners = dict(Partner.objects.filter(billing_order_id__gt=0).
                               values_list('billing_order_id', 'id'))

        show_log_objects = []

        for key, show_count in show_log_items.iteritems():
            billing_order_id, event_date, national_version, pp = key
            partner_id = cached_partners.get(billing_order_id)

            if partner_id:
                show_log_object = ShowLog(
                    eventdate=event_date,
                    partner_id=partner_id,
                    billing_order_id=billing_order_id,
                    national_version=national_version,
                    pp=pp,
                    show_count=show_count
                )
                show_log_objects.append(show_log_object)

        log.info('Delete records greater or equal: %s', left_date)
        ShowLog.objects.filter(eventdate__gte=left_date).delete()

        log.info('Save %d records', len(show_log_objects))
        ShowLog.objects.bulk_create(show_log_objects, batch_size=1000)

    # BEGIN main()
    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-d', '--days', dest='days', type='int', help='number of days to import', default=2)

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)

        now = datetime.now().date()
        left_date = now - timedelta(days=options.days - 1)

        log.info('Left date: %s', left_date)

        show_log_items = defaultdict(int)

        tables = safe_tables_for_daterange(yt, SHOW_LOG_PATH + '/1d', left_date, now)
        tables += safe_tables_for_daterange(yt, SHOW_LOG_PATH + '/stream/5min', now, now, '%Y-%m-%dT%H:%M:%S')

        for table in tables:
            for record in yt.read_table(table, format=yt.JsonFormat(), raw=False):
                try:
                    datasource_id = record.get('datasource_id')
                    billing_order_id = record.get('billing_order_id')

                    # Если нет BILLING_ORDER_ID, то делаем его из DATASOURCE_ID
                    if not billing_order_id and datasource_id:
                        billing_order_id = int(datasource_id) + 1

                    if not billing_order_id:
                        log.info('BILLING_ORDER_ID or DATASOURCE_ID not found')
                        continue

                    billing_order_id = int(billing_order_id)
                    event_date = datetime.strptime(record['iso_eventtime'][:10], '%Y-%m-%d')

                    if '/stream/' in table and event_date.date() != now:
                        log.info('%s: ignore %s', table, record['iso_eventtime'])
                        continue

                    national_version = record.get('national')
                    pp = int(record['pp'])
                except Exception:
                    log.info('Error while parse string')
                    continue

                # RASPTICKETS-7681: При импорте данных из баланса интерпретировать 601 pp как десктоп и сохранять его в базу как десктопный pp
                if pp == 601:
                    pp = 502

                record_key = (billing_order_id, event_date, national_version, pp)

                show_log_items[record_key] += 1

        write_showlog(show_log_items, left_date)

    except Exception:
        log.exception('Error:')
        sys.exit(1)

    log.info('Done')
