# -*- coding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

from HTMLParser import HTMLParser
import logging
import requests
import sys
from datetime import datetime

from optparse import OptionParser
from requests.auth import HTTPBasicAuth

from django.db import transaction
from django.conf import settings
from django.core.validators import URLValidator
from django.core.exceptions import ValidationError
from django.utils.html import strip_tags

from travel.avia.admin.lib.feeds import BadCompanyIATA, cache_codes, company_by_code, make_points
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.text import check_dirty_lang
from travel.avia.library.python.avia_data.models.review import FlightReview, FlightReviewSource


log = logging.getLogger(__name__)
create_current_file_run_log()

URL = 'http://topairlines.ru/api/reviews/'
LOGIN = 'yandex@localhost'
PASSWORD = settings.TOPAIRLINES_PASSWORD
SOURCE, _SC = FlightReviewSource.objects.get_or_create(code='topairlines.ru')


def get_reviews():
    reviews = []
    next_page = URL + "?format=json"

    while next_page is not None:
        log.info('Get: %s', next_page)
        r = requests.get(next_page, auth=HTTPBasicAuth(LOGIN, PASSWORD))

        if r.status_code != 200:
            log.warning('Bad status code: %r', r.status_code)
            break

        reviews_page = r.json()
        results = reviews_page.get('results', [])
        reviews.extend(results)

        next_page = reviews_page.get('next')

    return reviews


@transaction.atomic
def store_reviews(reviews, code_cache):
    for review in reviews:
        store_review(review, code_cache)


def store_review(review, code_cache):
    review_id = review.get('id')

    airline_code = review.get('airline').strip()
    if not airline_code:
        return

    try:
        airline = company_by_code(airline_code, log)

    except BadCompanyIATA:
        log.error("Bad code: %s", airline_code)
        return

    author_name = strip_tags(review.get('author') or "")

    from_iata = review.get('airport_from')
    to_iata = review.get('airport_to')

    from_point = code_cache.get(from_iata)
    to_point = code_cache.get(to_iata)

    if not all([from_point, to_point]):
        return

    from_settlement, from_station = make_points(from_point)
    to_settlement, to_station = make_points(to_point)

    full_direction = all([
        from_settlement,
        to_settlement,
        from_station,
        to_station
    ])

    if not full_direction:
        log.info('not full_direction: %s-%s', from_iata, to_iata)
        return

    try:
        created = review.get('created').split(".")[0].rstrip("Z")
        review_datetime = datetime.strptime(created, '%Y-%m-%dT%H:%M:%S')
    except ValueError:
        log.error("Bad datetime format: %s", created)
        return

    rating = int(float(review.get('rating')))
    review_content = HTMLParser().unescape(
        strip_tags(review.get('text') or "")
    )

    flight_date = datetime.strptime(review.get('date'), '%Y-%m-%d').date()

    try:
        dirty_lang_class = check_dirty_lang(review_content)

    except Exception:
        log.exception("ERROR")
        return

    except:
        log.error('Error while dirty lang check')
        return

    if dirty_lang_class:
        log.info('%s: %s' % (dirty_lang_class, review_content))

    review_url = 'http://topairlines.ru/reviews/{review_id}/'.format(review_id=review_id)
    try:
        URLValidator(review_url)

    except ValidationError:
        log.error('Bad url %s', review_url)
        return

    FlightReview.objects.get_or_create(
        review_id=review_id,
        source=SOURCE,
        defaults={
            'review_content': review_content,
            'review_datetime': review_datetime,
            'review_url': review_url,
            'flight_date': flight_date,
            'author_name': author_name,
            'settlement_from': from_settlement,
            'settlement_to': to_settlement,
            'station_from': from_station,
            'station_to': to_station,
            'airline': airline,
            'rating': rating,
            'dirty_lang_class': dirty_lang_class
        }
    )


def main():
    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('*** Start')

    if not SOURCE.enabled:
        log.info('%s disabled' % SOURCE)
        sys.exit(0)

    log.info('Load IATA/ICAO codes')
    code_cache = cache_codes()
    log.info('%s codes loaded', len(code_cache))

    try:
        reviews = get_reviews()
        store_reviews(reviews, code_cache)
        log.info('Stored: %d', len(reviews))

    except Exception:
        log.exception('ERROR')
        return

    log.info('*** Done')
