# -*- encoding: utf-8 -*-
import functools
import os
import sys
from datetime import datetime, timedelta
from optparse import OptionParser

from xml.etree.ElementTree import Element, SubElement, tostring
import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

from travel.avia.admin.lib.s3sync import s3sync

RETURN_TRIP = False
RECEPIENTS = ['avia-process@yandex-team.ru',
              'bfa@yandex-team.ru']
NATIONAL_TO_LANG = {'ua': 'uk'}
NATIONAL_TO_CURR = {'ru': 'RUR', 'ua': 'UAH'}


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.conf import settings
    from django.core.mail.message import EmailMultiAlternatives

    from travel.avia.library.python.common.models.geo import Country, CodeSystem, Station, Settlement
    from travel.avia.library.python.common.models.transport import TransportType
    from travel.avia.library.python.avia_data.models import AviaRecipe
    from travel.avia.library.python.avia_data.models import MinPrice

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import yt_last_logs_tables, configure_wrapper

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    def write_elements_to_file(file_name, elements):
        xml = tostring(elements, 'utf-8')

        file_name = os.path.join(settings.MEDIA_ROOT, 'avia', 'files', file_name)
        tmp_file_name = '%s.tmp' % file_name

        file_dir = os.path.dirname(tmp_file_name)
        if not os.path.exists(file_dir):
            os.makedirs(file_dir)

        with open(tmp_file_name, 'w') as f:
            f.write('<?xml version="1.0" encoding="utf-8"?>\n')
            f.write(xml)
            f.flush()
            os.fsync(f)
            f.close()

        os.rename(tmp_file_name, file_name)

        log.info('Saved to %s', file_name)

    def make_references():
        # Рецепты
        recipes = Element('recipes', {'date': datetime.now().strftime('%Y-%m-%d')})
        for r in AviaRecipe.objects.all():
            if r.enabled_ru or r.enabled_ua or r.enabled_tr or r.enabled_com:
                recipe_data = {
                    'id': str(r.id),
                    'title_ru': r.title_ru if r.title_ru else '',
                    'title_uk': r.title_uk if r.title_uk else '',
                    'title_tr': r.title_tr if r.title_tr else '',
                    'title_en': r.title_en if r.title_en else '',
                }

                recipe = SubElement(recipes, 'recipe', recipe_data)

                recipe_countries = r.countries.all()
                recipe_settlements = r.settlements.all()

                if recipe_countries:
                    countries = SubElement(recipe, 'countries')

                    for c in recipe_countries:
                        country_data = {
                            'id': str(c.id)
                        }

                        SubElement(countries, 'country', country_data)

                if recipe_settlements:
                    settlements = SubElement(recipe, 'settlements')
                    for s in r.settlements.all():
                        settlement_data = {
                            'id': str(s.id)
                        }
                        SubElement(settlements, 'settlement', settlement_data)

        write_elements_to_file('recipes.xml', recipes)

        points = Element('points', {'date': datetime.now().strftime('%Y-%m-%d')})

        countries = SubElement(points, 'countries')

        for c in Country.objects.filter(code__isnull=False):
            l_title = c.new_L_title
            country_data = {
                'code': c.code,
                'title_ru': l_title.ru_nominative or '',
                'title_uk': l_title.uk_nominative or '',
                'title_tr': l_title.tr_nominative or '',
                'title_en': l_title.en_nominative or '',
                'title_de': l_title.de_nominative or '',
            }

            SubElement(countries, 'country', country_data)

        settlements = SubElement(points, 'settlements')

        iata_codesystem = CodeSystem.objects.get(code='iata')
        sirena_codesystem = CodeSystem.objects.get(code='sirena')

        for s in Settlement.objects.filter(hidden=False):
            airports = [a for a in s.station_set.filter(t_type_id=TransportType.PLANE_ID)]
            iata_codes = [a.iata for a in airports if a.iata]
            sirena_codes = [a.sirena_id for a in airports if a.sirena_id]

            if airports:
                l_title = s.new_L_title

                settlement_data = {
                    'id': str(s.id),
                    'title': s.title if s.title else '',
                    'title_ru': l_title.ru_nominative or '',
                    'title_uk': l_title.uk_nominative or '',
                    'title_tr': l_title.tr_nominative or '',
                    'title_en': l_title.en_nominative or '',
                    'title_de': l_title.de_nominative or '',
                    'country_code': s.country.code if (s.country and s.country.code) else '',
                }

                if s.iata or len(iata_codes) == 1:
                    settlement_data['iata'] = s.iata or iata_codes[0]

                if s.sirena_id or len(sirena_codes) == 1:
                    settlement_data['sirena'] = s.sirena_id or sirena_codes[0]

                settlement = SubElement(settlements, 'settlement', settlement_data)

                for a in airports:
                    a_l_title = a.new_L_title
                    station_data = {
                        'id': str(a.id),
                        'title': a.title if a.title else '',
                        'title_ru': a_l_title.ru_nominative or '',
                        'title_uk': a_l_title.uk_nominative or '',
                        'title_tr': a_l_title.tr_nominative or '',
                        'title_en': a_l_title.en_nominative or '',
                        'title_de': a_l_title.de_nominative or '',
                    }

                    iata = a.get_code(iata_codesystem)
                    sirena = a.get_code(sirena_codesystem)

                    if iata:
                        station_data['iata'] = iata

                    if sirena:
                        station_data['sirena'] = sirena

                    SubElement(settlement, 'station', station_data)

        write_elements_to_file('points.xml', points)

    def get_settlements_by_id():
        log.info('Load settlements')
        return {s.id: s for s in Settlement.objects.all()}

    def get_station_map():
        log.info('Load stations')
        stations = Station.objects.filter(
            t_type_id=TransportType.PLANE_ID,
            settlement_id__isnull=False,
            hidden=False
        )

        stations = [s for s in stations if s.iata or s.sirena_id]

        return {s.point_key: s.settlement.point_key for s in stations}

    def reduce_directions(key, records):
        """
        Reduce popular directions
        """
        yield {
            'direction': key['direction'],
            'count': sum([int(r['count']) for r in records])
        }

    def map_directions(stations_map, national_version, record):
        """
        Map popular directions
        """
        def fix_settlement(point_key):
            if point_key.startswith('s'):
                return stations_map.get(point_key)

            return point_key

        try:
            from_id = fix_settlement(record['fromId'])
            to_id = fix_settlement(record['toId'])

        except Exception:
            return

        useful_search = (
            from_id and
            to_id and
            record.get('national_version') == national_version
        )

        if useful_search:
            yield {
                'direction': '%s_%s' % (from_id, to_id),
                'count': 1
            }

    def min_prices(top_directions, national_version):
        today = datetime.now().date()
        left_date_border = today + timedelta(days=30)

        for direction in top_directions:
            from_key, to_key = direction.split('_')

            min_price = MinPrice.objects.filter(
                departure_settlement_id=from_key[1:],
                arrival_settlement=to_key[1:],
                date_forward__gte=today,
                date_forward__lte=left_date_border,
                date_backward__isnull=True,
                passengers=MinPrice.passengers_key_from_dict({'adults': 1}),
                national_version=national_version,
                price__gte=999,
                price__lte=39999
            ).order_by('price')[:1]

            if not min_price:
                continue

            yield min_price[0]

    def make_file(min_prices, cities, output_file, national_version, landing_page=False):
        lang = NATIONAL_TO_LANG.get(national_version, 'ru')

        yml_catalog = Element('yml_catalog', {'date': datetime.now().strftime('%Y-%m-%d')})

        shop = SubElement(yml_catalog, 'shop')

        shop_name = SubElement(shop, 'name')
        shop_name.text = 'avia.yandex.ru'

        shop_company = SubElement(shop, 'company')
        shop_company.text = 'avia.yandex.ru'

        shop_url = SubElement(shop, 'url')
        shop_url.text = 'avia.yandex.ru'

        currencies = SubElement(shop, 'currencies')
        SubElement(currencies, 'currency', {
            'id': NATIONAL_TO_CURR.get(national_version),
            'rate': '1'
        })

        categories = SubElement(shop, 'categories')
        category = SubElement(categories, 'category', {'id': "1"})
        category.text = u'Aviadirections'

        offers = SubElement(shop, 'offers')

        for min_price in min_prices:

            avia_host = 'avia.yandex.%s' % national_version

            if landing_page:
                try:
                    min_price_url = 'https://%s/routes/%d/%d/' % (
                        avia_host,
                        min_price.departure_settlement_id,
                        min_price.arrival_settlement_id,
                    )
                except ValueError:
                    continue

            else:
                min_price_url = 'https://%s/search/?fromId=c%d&toId=c%d&adult_seats=1&when=%s' % (
                    avia_host,
                    min_price.departure_settlement_id,
                    min_price.arrival_settlement_id,
                    min_price.date_forward,
                )

            offer_id = '%d%d' % (
                min_price.departure_settlement_id,
                min_price.arrival_settlement_id,
            )

            offer = SubElement(offers, 'offer', {'id': offer_id, 'available': 'true'})

            url = SubElement(offer, 'url')
            url.text = min_price_url

            price = SubElement(offer, 'price')
            price.text = str(int(min_price.price / 100) * 100)

            currencyId = SubElement(offer, 'currencyId')
            currencyId.text = min_price.currency.code

            categoryId = SubElement(offer, 'categoryId')
            categoryId.text = '1'

            offer_name = SubElement(offer, 'name')
            offer_name.text = 'c%d_c%d' % (
                min_price.departure_settlement_id,
                min_price.arrival_settlement_id,
            )

            param_name_from = SubElement(offer, 'param', {'name': 'fromName'})
            param_name_from.text = cities.get(min_price.departure_settlement_id).L_title(lang=lang)
            param_name_to = SubElement(offer, 'param', {'name': 'toName'})
            param_name_to.text = cities.get(min_price.arrival_settlement_id).L_title(lang=lang)

        write_elements_to_file('%s.xml' % output_file, yml_catalog)

    # BEGIN main()
    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-n", "--national_version", dest="national_version", default='ru')

    options, args = optparser.parse_args()

    if options.verbose:
        from travel.avia.admin.lib.s3sync import log as s3sync_log
        add_stdout_handler(log)
        add_stdout_handler(s3sync_log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        configure_wrapper(yt)
        # Фиды стран, рецептов, городов
        make_references()

        stations_map = get_station_map()

        tmp_table = yt.create_temp_table()
        source_tables = [
            yt.TablePath(table, columns=['fromId', 'toId', 'national_version'])
            for table in yt_last_logs_tables('//home/avia/logs/avia-users-search-log', 1)
        ]

        log.info('Build top directions')
        yt.run_map_reduce(
            functools.partial(map_directions, stations_map, options.national_version),
            source_table=source_tables,
            destination_table=tmp_table,
            reducer=reduce_directions,
            reduce_by=['direction'],
        )

        top_direction = []
        for record in yt.read_table(tmp_table, raw=False, format=yt.JsonFormat()):
            direction = record['direction']
            count = int(record['count'])
            top_direction.append((count, direction))

        sorted_top_directions = [t[1] for t in sorted(top_direction, reverse=True)]

        top_directions_head = list(min_prices(sorted_top_directions[:100], options.national_version))
        top_directions_tail = list(min_prices(sorted_top_directions[100:1250], options.national_version))
        top_directions_1250_3000 = list(min_prices(sorted_top_directions[1250:3000], options.national_version))

        file_suffix = '' if options.national_version == 'ru' else '_' + options.national_version
        cities = get_settlements_by_id()

        make_file(top_directions_head, cities, 'adwords_head' + file_suffix, options.national_version)
        make_file(top_directions_tail, cities, 'adwords_tail' + file_suffix, options.national_version)
        make_file(top_directions_1250_3000, cities, 'adwords_1250_3000' + file_suffix, options.national_version)

        make_file(top_directions_head, cities, 'adwords_head_landing' + file_suffix, options.national_version, landing_page=True)
        make_file(top_directions_tail, cities, 'adwords_tail_landing' + file_suffix, options.national_version, landing_page=True)
        make_file(top_directions_1250_3000, cities, 'adwords_1250_3000_landing' + file_suffix, options.national_version, landing_page=True)

        s3sync('adwords')
        s3sync('adwords_{}'.format(options.national_version))

        current_env = settings.ENVIRONMENT

        if current_env == 'production':
            body = 'https://avia.yandex.ru/files/adwords_head%s.xml\n' % file_suffix
            body += 'https://avia.yandex.ru/files/adwords_tail%s.xml\n' % file_suffix
            body += 'https://avia.yandex.ru/files/adwords_1250_3000%s.xml\n' % file_suffix
            body += 'https://avia.yandex.ru/files/adwords_head_landing%s.xml\n' % file_suffix
            body += 'https://avia.yandex.ru/files/adwords_tail_landing%s.xml\n' % file_suffix
            body += 'https://avia.yandex.ru/files/adwords_1250_3000_landing%s.xml\n' % file_suffix

            mail = EmailMultiAlternatives(
                subject='New adwords.xml was born',
                body=body,
                from_email=settings.SERVER_EMAIL,
                to=RECEPIENTS,
            )

            mail.send()

    except Exception as e:
        log.exception('Error: {}'.format(e))
        sys.exit(1)

    log.info('Done')
