# -*- encoding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
import os
from datetime import datetime, timedelta
from collections import namedtuple
from optparse import OptionParser

from django import setup
from django.conf import settings
from django.core.mail import send_mail
from django.db.models import Q
from library.python import resource
from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.admin.lib.logs import create_current_file_run_log
from travel.avia.admin.lib.yql_helpers import get_yql_operation_url, log_errors

ALLOWED_ENVS = ['production', 'sandbox_production']
REDIRECT_QUERY_FILE = os.path.join('resfs/file', os.path.dirname(__file__), 'data', 'top_avia_companies_on_front.sql')
_RECIPIENT_LIST = ['avia-quality@yandex-team.ru']


def do_query(logger, yql_client, left_date, right_date):
    query = resource.find(REDIRECT_QUERY_FILE)

    r = yql_client.query(query,  syntax_version=1)
    r.run(
        parameters=ValueBuilder.build_json_map({
            '$start_date': ValueBuilder.make_string(left_date.strftime('%Y-%m-%d')),
            '$end_date': ValueBuilder.make_string(right_date.strftime('%Y-%m-%d')),
        }),
    )
    logger.info('YQL operation: %s', get_yql_operation_url(r))

    return r


def prepare_message(company_by_code, companies_on_front, max_records):
    message_data = []
    company_codes = [company for company in companies_on_front if company['code'] in company_by_code]
    for company_idx, company in enumerate(company_codes[:max_records], start=1):
        title = company_by_code[company['code']].title
        company_id = company_by_code[company['code']].id

        message_data.append(
            u'{}. Компания {} ({}), число показов: {}. https://admin.avia.yandex-team.ru/admin/www/company/{}'.format(
                company_idx, title, company['code'], company['count'], company_id)
        )

    return '\n'.join(message_data)


def get_avia_companies_from_logs(logger, left_date, right_date):
    yql_client = YqlClient(token=settings.YQL_TOKEN)

    yql_query = do_query(logger, yql_client, left_date, right_date)

    yql_query.wait_progress()
    companies_on_front = []
    if yql_query.is_ok:
        query_result = yql_query.get_results()
        logger.info('YQL query is done')
    else:
        log_errors(yql_query, logger)
        return companies_on_front

    for table in query_result:
        table.fetch_full_data()
        for row in table.rows:
            companies_on_front.append({'count': row[0], 'code': row[1]})

    return companies_on_front


def get_companies_without_logo():
    setup()

    company_without_logo = Company.objects.filter(
        Q(sirena_id__isnull=False) | Q(iata__isnull=False),
        Q(svg_logo2__isnull=True) | Q(svg_logo2=''),
        t_type=TransportType.PLANE_ID
    ).values('id', 'iata', 'sirena_id', 'title')

    CompanyInfo = namedtuple("CompanyInfo", ('id', 'title'))

    company_codes = dict()
    for company in company_without_logo:
        company_codes[company['iata']] = CompanyInfo(company['id'], company['title'])
        company_codes[company['sirena_id']] = CompanyInfo(company['id'], company['title'])

    return company_codes


def main():
    logger = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser()
    optparser.add_option('-d', '--days', dest='days', type='int', default=7)
    optparser.add_option('-m', '--max_records', dest='max_records', type='int',
                         help='max count of records companies without logo', default=10)

    options, args = optparser.parse_args()

    right_date = datetime.now().date()
    left_date = right_date - timedelta(days=options.days)

    if os.getenv('YANDEX_ENVIRONMENT_TYPE') not in ALLOWED_ENVS:
        return

    logger.info('Start')

    companies_on_front = get_avia_companies_from_logs(logger, left_date, right_date)

    company_by_code = get_companies_without_logo()

    message = prepare_message(company_by_code, companies_on_front, options.max_records)

    if not message:
        logger.info('Companies without a logo were not found. End')
        return

    message_title = 'Top {} companies without logo for the days from {} to {}'.format(
        options.max_records, left_date, right_date
    )

    send_mail(message_title, message=message,
              from_email=settings.SERVER_EMAIL, recipient_list=_RECIPIENT_LIST)

    logger.info('End')
