# -*- encoding: utf-8 -*-
# Отчёты из этого скрипта не использутся в STAT

import travel.avia.admin.init_project  # noqa

import argparse
import logging
import os
from collections import defaultdict
from datetime import datetime, timedelta

from django.conf import settings
from library.python import resource
import yt.wrapper as yt
from yql.api.v1.client import YqlClient

from travel.avia.library.python.common.models.partner import Partner

from travel.avia.admin.lib.logs import create_current_file_run_log, add_stdout_handler
from travel.avia.admin.lib.statface_helpers import post_data_to_stat
from travel.avia.admin.lib.yql_helpers import get_yql_operation_url, log_errors
from travel.avia.admin.lib.yt_helpers import yt_client_fabric


logger = logging.getLogger(__name__)
create_current_file_run_log()


ALLOWED_ENVS = ['production', 'dev']
REDIRECT_QUERY_FILE = os.path.join('resfs/file', os.path.dirname(__file__), 'data', 'redirect_stat.sql')
REDIRECT_ERROR_QUERY_FILE = os.path.join('resfs/file', os.path.dirname(__file__), 'data', 'redirect_error_stat.sql')
STAT_NAME = 'ticket.yandex/Redirects/ByPartner'
STAT_REDIRECT_ERROR_NAME = 'ticket.yandex/Redirects/RedirectError'

_DATE_FORMAT = '%Y-%m-%d'
_ONE_DAY = timedelta(days=1)


def _parse_date(datestr, default=datetime.today()):
    return datetime.strptime(datestr, '%Y-%m-%d').date() if datestr else datetime.today()


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', dest='verbose',  action='store_true')
    parser.add_argument('--yesterday', action='store_true')
    parser.add_argument('--left-date', type=_parse_date, default=datetime.today().date())
    parser.add_argument('--right-date', type=_parse_date, default=datetime.today().date())
    parser.add_argument('--skip-env-check', action='store_true')

    return parser.parse_args()


def send_redirect_stat(yt_client, table):
    logger.info('Start sending redirects stat to Statface')
    beta = settings.ENVIRONMENT != 'production'
    data = list(yt_client.read_table(table, format=yt.JsonFormat()))
    post_data_to_stat(STAT_NAME, data, scale='h', beta=beta)
    logger.info('Upload table %s succeed', table)

    aggregated = defaultdict(int)
    for record in data:
        key = record['fielddate'], record['billing_client_id']
        aggregated[key] += record['n_redirects']

    return aggregated


def do_redirect_stat_query(yql_client, left_date, right_date, output_table):
    query = resource.find(REDIRECT_QUERY_FILE)

    argument_string = '\n'.join([
        '$start_date="{}";'.format(left_date.strftime(_DATE_FORMAT)),
        '$end_date="{}";'.format(right_date.strftime(_DATE_FORMAT)),
        '$output_path="{}";'.format(output_table),
    ])
    current_query = '{arguments}\n{query}'.format(
        arguments=argument_string,
        query=query,
    )

    r = yql_client.query(current_query)
    r.run()
    logger.info('YQL operation: %s', get_yql_operation_url(r))

    return r


def do_redirect_error_query(yql_client, left_date, right_date, output_table):
    logger.info('Start redirect error query')
    query = resource.find(REDIRECT_ERROR_QUERY_FILE)

    argument_string = '\n'.join([
        '$start_date="{}";'.format(left_date.strftime(_DATE_FORMAT)),
        '$end_date="{}";'.format(right_date.strftime(_DATE_FORMAT)),
        '$output_path="{}";'.format(output_table),
    ])
    current_query = '{arguments}\n{query}'.format(
        arguments=argument_string,
        query=query,
    )

    r = yql_client.query(current_query)
    r.run()
    logger.info('YQL operation: %s', get_yql_operation_url(r))

    return r


def main():
    args = parse_args()
    if args.verbose:
        add_stdout_handler(logger)

    logger.info('Start')

    if args.yesterday:
        left_date = right_date = datetime.today().date() - _ONE_DAY
    else:
        left_date = args.left_date
        right_date = args.right_date

    if not (args.skip_env_check or settings.ENVIRONMENT in ALLOWED_ENVS):
        logger.info('Can work only in %s', ', '.join(ALLOWED_ENVS))
        return

    yql_client = YqlClient(token=settings.YQL_TOKEN)
    yt_client = yt_client_fabric.create()

    current_table = yt_client.create_temp_table()
    logger.info('Redirect stat table: %s', current_table)
    redirect_yql_query = do_redirect_stat_query(yql_client, left_date, right_date, current_table)

    output_error_table = yt_client.create_temp_table()
    logger.info('Redirect error table: %s', output_error_table)
    redirect_error_yql_query = do_redirect_error_query(yql_client, left_date, right_date, output_error_table)

    partners = dict(Partner.objects.values_list('code', 'billing_client_id'))

    redirect_yql_query.wait_progress()
    if redirect_yql_query.is_ok:
        redirect_stat = send_redirect_stat(yt_client, current_table)
    else:
        log_errors(redirect_yql_query, logger)
        redirect_stat = {}

    redirect_error_yql_query.wait_progress()
    if redirect_yql_query.is_ok:
        redirect_error_stat = []
        for record in yt_client.read_table(output_error_table, format=yt.JsonFormat()):
            partner = record['partner'].strip()
            if partner.startswith('dohop'):
                partner = 'dohop'
            if not partner:
                record['partner'] = '__empty__partner__'
            billing_client_id = partners.get(partner)

            if billing_client_id is None:
                logger.error('Billing_client_id was not found for partner  with code  %s', record['partner'])
            else:
                redirect_count = redirect_stat.get((record['fielddate'], billing_client_id), 0)
                record['redirect_count'] = redirect_count
                if redirect_count != 0:
                    record['ratio'] = float(record['count']) / (redirect_count + record['count'])

            redirect_error_stat.append(record)

        logger.info('Ready to send %d record about redirect errors', len(redirect_error_stat))
        if redirect_error_stat:
            response = post_data_to_stat(
                STAT_REDIRECT_ERROR_NAME,
                redirect_error_stat,
                scale='h',
                beta=settings.ENVIRONMENT != 'production',
            )
            if response.status_code != 200:
                logger.error(
                    'Upload redirect error table failed. Code: %d. Message: %s',
                    response.status_code, response.content
                )
        else:
            logger.info('Nothing to send')

    else:
        log_errors(redirect_yql_query, logger)

    logger.info('End')
