import travel.avia.admin.init_project  # noqa

import argparse
import logging
from collections import defaultdict
from itertools import chain
from datetime import datetime, timedelta

from django.conf import settings

from travel.avia.admin.lib.statface_helpers import post_data_to_stat
from travel.avia.admin.lib.yt_helpers import yt_client_fabric, safe_tables_for_daterange
from travel.avia.admin.lib.logs import create_current_file_run_log, add_stdout_handler

import yt.wrapper as yt


_ONE_HOUR = 60 * 60
_ONE_DAY = timedelta(days=1)

YT_LOG_DIR = '//home/logfeller/logs/avia-revise-log'
STAT_NAME = 'ticket.yandex/Revise/PartnerStat'

ALLOWED_ENVS = ['production', 'dev']


logger = logging.getLogger(__name__)


def compute_revise_stat(records, sub_key_column, group_columns):
    def get_key(record):
        return tuple(record[column] for column in group_columns)

    stat = defaultdict(lambda: defaultdict(int))
    for record in records:
        record['unixtime'] = record['unixtime'] - record['unixtime'] % _ONE_HOUR
        key = get_key(record)
        stat[key][record[sub_key_column]] += 1

    for key, group_stat in stat.iteritems():
        group_sum = sum(group_stat.itervalues())
        for result, count in group_stat.iteritems():
            yield dict(
                zip(group_columns, key),
                result=result,
                count=count,
                rate=float(count) / group_sum,
            )


def _parse_date(datestr, default=datetime.today()):
    return datetime.strptime(datestr, '%Y-%m-%d').date() if datestr else datetime.today()


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', dest='verbose',  action='store_true')
    parser.add_argument('--yesterday', action='store_true')
    parser.add_argument('--left-date', type=_parse_date, default=datetime.today().date())
    parser.add_argument('--right-date', type=_parse_date, default=datetime.today().date())
    parser.add_argument('--skip-env-check', action='store_true')

    return parser.parse_args()


def main():
    args = parse_args()
    if args.verbose:
        add_stdout_handler(logger)

    logger.info('Start')
    create_current_file_run_log()

    if args.yesterday:
        left_date = right_date = (datetime.today() - _ONE_DAY).date()
    else:
        left_date = args.left_date
        right_date = args.right_date

    if left_date == right_date == datetime.today().date():
        revise_log_path = YT_LOG_DIR + '/stream/5min'
        date_format = '%Y-%m-%dT%H:%M:%S'
    else:
        revise_log_path = YT_LOG_DIR + '/1d'
        date_format = '%Y-%m-%d'

    if not (args.skip_env_check or settings.ENVIRONMENT in ALLOWED_ENVS):
        logger.info('Can work only in %s', ', '.join(ALLOWED_ENVS))
        return

    yt_client = yt_client_fabric.create()

    logger.info('Left date: %s', left_date.strftime('%Y-%m-%d'))
    logger.info('Right date: %s', right_date.strftime('%Y-%m-%d'))
    logger.info('Table date format: %s', date_format)
    logger.info('Revise log path: %s', revise_log_path)

    tables = [
        yt.TablePath(table, columns=['unixtime', 'partner', 'result'])
        for table in safe_tables_for_daterange(yt_client, revise_log_path, left_date, right_date, date_format)
    ]

    revise_records = chain(*(yt_client.read_table(table, format=yt.JsonFormat()) for table in tables))

    records = []
    for record in compute_revise_stat(revise_records, 'result', ['unixtime', 'partner']):
        record['fielddate'] = datetime.strftime(
            datetime.fromtimestamp(record.pop('unixtime')),
            '%Y-%m-%d %H:%M:%S',
        )

        records.append(record)

    logger.info('Ready to send %d records', len(records))
    if not records:
        logger.info('Nothing to send')
    else:
        post_data_to_stat(STAT_NAME, records, beta=settings.ENVIRONMENT != 'production', scale='h')
        logger.info('Upload table succeed')

    logger.info('End')
