# -*- encoding: utf-8 -*-
import functools
import logging
from datetime import datetime
from itertools import chain
from optparse import OptionParser


from travel.avia.admin.lib.logs import add_stdout_handler
from travel.avia.admin.lib.yt_helpers import AviaYtClientFabric

log = logging.getLogger(__name__)

RUSSIA_ID = 'l225'

GOOD_PARTNERS = {
    'agent',
    'anywayanyday',
    'aviakass',
    'aviaoperator',
    'biletdv',
    'biletikaeroag',
    'biletinet',
    'biletix',
    'booktripruag',
    'citytravel',
    'citytravel1',
    'clickaviatravelab',
    'expressavia',
    'kiwi',
    'kupibilet',
    'megotravel',
    'nebotravel',
    'ozon',
    'superkassa',
    'supersaver',
    'svyaznoy',
    'ticketsru',
    'tripcom',
    'trip_ru',
    'uzairways',
}


def zero_redirect_price(good_partners, russian_geo_objects, source_row):
    if (source_row['national_version'] == 'ru' and source_row['partnerCode'] in good_partners and
            (source_row['fromId'] not in russian_geo_objects or source_row['toId'] not in russian_geo_objects)):
        source_row['price'] = 0.

    yield source_row


def main():
    import travel.avia.admin.init_project  # noqa

    from django.conf import settings

    if settings.YANDEX_ENVIRONMENT_TYPE != 'production':
        log.warn('Ony production environment allowed')
        return

    optparser = OptionParser()
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-s', '--source', dest='source', help='Redirect log source folder')
    optparser.add_option('-d', '--destination', dest='destination', help='Destination folder')
    optparser.add_option('-f', '--from-date', dest='from_date', help='From date (YYYY-MM-DD)')
    optparser.add_option('-t', '--to-date', dest='to_date', help='To date (YYYY-MM-DD)')
    optparser.add_option('-b', '--backup', dest='backup', help='Backup folder')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    if not options.source:
        optparser.print_help()
        optparser.error('Source required')

    if not options.destination:
        optparser.print_help()
        optparser.error('Destination required')

    if not options.from_date:
        optparser.print_help()
        optparser.error('From date required')
    from_date = datetime.strptime(options.from_date, '%Y-%m-%d').date()

    if not options.to_date:
        optparser.print_help()
        optparser.error('To date required')
    to_date = datetime.strptime(options.to_date, '%Y-%m-%d').date()

    ytc = AviaYtClientFabric().create()

    if not ytc.exists(options.source):
        optparser.error('Unknown source: {}'.format(options.source))

    if ytc.exists(options.destination):
        optparser.error('Destination already exists: {}'.format(options.destination))
    else:
        ytc.create('map_node', options.destination, recursive=True)
        log.info('Destination folder created: %s', options.destination)

    if options.backup and not ytc.exists(options.backup):
        ytc.create('map_node', options.backup, recursive=True)
        log.info('Backup folder created: %s', options.backup)

    log.info('Start')
    log.info('Source: %s', options.source)
    log.info('Destination: %s', options.destination)
    log.info('From: %s', from_date.strftime('%Y-%m-%d'))
    log.info('To: %s', to_date.strftime('%Y-%m-%d'))

    log.info('Load russian geo objects for filter')
    russian_geo_objects = set()
    for row in chain(ytc.read_table('//home/rasp/reference/settlement'), ytc.read_table('//home/rasp/reference/station')):
        if row['country_id'] != RUSSIA_ID:
            continue
        russian_geo_objects.add(row['id'])
    log.info('Loaded %d russian geo objects', len(russian_geo_objects))

    for table in ytc.tables_for_daterange(options.source, from_date, to_date):
        with ytc.Transaction():
            ytc.lock(table, waitable=True)

            destination_table = '{}/{}'.format(options.destination, table.split('/')[-1])
            log.info('From %s to %s', table, destination_table)

            ytc.create('table', destination_table, attributes={
                'schema': ytc.get_attribute(table, 'schema'),
                'processed': ytc.get_attribute(table, 'processed'),
            })

            if options.backup:
                backup_table = '{}/{}'.format(options.backup, table.split('/')[-1])
                ytc.copy(table, backup_table)
                log.info('Backup %s', backup_table)

            ytc.run_map(
                functools.partial(zero_redirect_price, GOOD_PARTNERS, russian_geo_objects),
                source_table=table,
                destination_table=destination_table,
            )

    log.info('Done')
