# -*- encoding: utf-8 -*-
import logging
from datetime import datetime
from optparse import OptionParser

REDIR_LOG = '//home/avia/logs/avia-json-redir-log'


def _valid_source(utm_source, utm_medium, utm_content):
    # type: (str, str, str) -> bool
    if utm_source == 'rasp' and utm_medium == 'redirect':
        return False
    if utm_source == 'sovetnik' and utm_content == 'redirect':
        return False
    if utm_source == 'wizard_ru' or utm_source == 'unisearch_ru':
        return False

    return True


def main():
    import travel.avia.admin.init_project  # noqa

    from django.core.exceptions import ObjectDoesNotExist

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import AviaYtClientFabric
    from travel.avia.library.python.common.models.partner import Partner

    logger = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser(description=u'Установка нулевой цены за переход')
    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('--source', dest='source', default=REDIR_LOG)
    optparser.add_option('--partner', dest='partner', help=u'Код партнер')
    optparser.add_option('--amount', dest='amount', help=u'Количество переходов для зануления цены')
    optparser.add_option('--price-total', dest='price_total', help=u'Стоимость кликов для зануления')
    optparser.add_option('--destination', dest='destination', help=u'Путь для сохранения измененных логов')
    optparser.add_option('--backup-destination', dest='backup_destination', help=u'Путь для бекапа')
    optparser.add_option('--from-date', dest='from_date', help=u'Дата первого лога YYYY-MM-DD')
    optparser.add_option('--to-date', dest='to_date', help=u'Дата последнего лога YYYY-MM-DD')
    optparser.add_option('--all-sources', dest='all_sources', action='store_true', help=u'Занулять все редиректы независимо от источника')
    optparser.add_option('--all-national', dest='all_national', action='store_true', help=u'Занулять редиректы всех национальных версий, иначе только ru')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(logger)

    ytc = AviaYtClientFabric().create()

    try:
        from_date = datetime.strptime(options.from_date, '%Y-%m-%d').date()
        to_date = datetime.strptime(options.to_date, '%Y-%m-%d').date()

        destination = options.destination
        if not destination:
            raise ValueError('Empty destination')
        if ytc.exists(destination):
            raise ValueError('Destination "{}" exists'.format(destination))

        backup_destination = options.backup_destination
        if not backup_destination:
            raise ValueError('Empty backup_destination')
        if ytc.exists(backup_destination):
            raise ValueError('Destination "{}" exists'.format(backup_destination))

        if backup_destination == destination:
            raise ValueError('Destination and backup destination must by different')

        amount = int(options.amount) if options.amount else 0
        price_total = float(options.price_total) if options.price_total else 0.
        if amount <= 0 and price_total <= 0:
            raise ValueError('Set amount or price-total')

        try:
            partner = Partner.objects.get(code=options.partner)
        except ObjectDoesNotExist:
            raise ValueError('Unknown partner "{}"'.format(options.partner))

    except (ValueError, TypeError) as e:
        optparser.print_help()
        return optparser.error(e)

    logger.info('Start')
    logger.info('Source: %s', options.source)
    logger.info('Destination: %s', destination)
    logger.info('Backup: %s', backup_destination)
    logger.info('From: %s', from_date.strftime('%Y-%m-%d'))
    logger.info('To: %s', to_date.strftime('%Y-%m-%d'))
    logger.info('Partner: %s', partner.code)
    logger.info('Amount: %s', amount)
    logger.info('Price total: %s', price_total)

    with ytc.Transaction():
        ytc.lock(options.source, waitable=True)
        ytc.create('map_node', destination, recursive=True)
        logger.info('%s created', destination)
        ytc.create('map_node', backup_destination, recursive=True)
        logger.info('%s created', backup_destination)

        count = 0
        price = 0.
        for table in ytc.tables_for_daterange(options.source, from_date, to_date):
            if 0 < amount <= count:
                break
            if 0 < price_total <= price:
                break

            ytc.lock(table, waitable=True)

            destination_table = '{}/{}'.format(destination, table.split('/')[-1])

            backup_table = '{}/{}'.format(backup_destination, table.split('/')[-1])
            ytc.copy(table, backup_table)
            logger.info('Backup %s', backup_table)

            logger.info('From %s to %s', table, destination_table)
            ytc.create('table', destination_table, attributes={
                'schema': ytc.get_attribute(table, 'schema'),
                'processed': ytc.get_attribute(table, 'processed', default=[]),
                'compression_codec': ytc.get_attribute(table, 'compression_codec'),
                'optimize_for': ytc.get_attribute(table, 'optimize_for'),
            })

            logger.info('Processing %s (%d) ...', table, ytc.row_count(table))
            rows = []
            for row in ytc.read_table(table):
                if row['partnerCode'] != partner.code \
                        or (0 < amount <= count) \
                        or (0 < price_total <= price) \
                        or (not _valid_source(row['utm_source'], row['utm_medium'], row['utm_content']) and not options.all_sources) \
                        or (row['national_version'] != 'ru' and not options.all_national):
                    rows.append(row)
                    continue

                count += 1
                price += row['price']
                row['price'] = 0.
                rows.append(row)

            logger.info('Updated %d redirects, price %f', count, price)

            logger.info('Writing %s (%d) ...', destination_table, len(rows))
            ytc.write_table(destination_table, rows)

    logger.info('Done')
