# -*- encoding: utf-8 -*-
from __future__ import absolute_import, unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import argparse
import logging

import dateutil.parser

from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.yt_helpers import AviaYtClientFabric

logger = logging.getLogger(__name__)

REDIR_LOG = '//home/avia/logs/avia-json-redir-log'


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    parser.add_argument('--from-date', help='Дата первого лога YYYY-MM-DD', required=True,
                        type=lambda s: dateutil.parser.isoparse(s).date())
    parser.add_argument('--to-date', help='Дата последнего лога YYYY-MM-DD', required=True,
                        type=lambda s: dateutil.parser.isoparse(s).date())
    parser.add_argument('--destination', help='Путь для сохранения измененных логов', required=True)
    parser.add_argument('--backup-destination', help='Путь для бекапа', required=True)
    parser.add_argument('--user-ip-by-qid-table', help='Путь до таблицы сопоставления qid и user-ip', required=True)

    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(logger)
    create_current_file_run_log()

    ytc = AviaYtClientFabric().create()

    if ytc.exists(args.destination):
        parser.error('Destination "{}" exists'.format(args.destination))

    if ytc.exists(args.backup_destination):
        parser.error('Destination "{}" exists'.format(args.backup_destination))

    if args.backup_destination == args.destination:
        parser.error('Destination and backup destination must by different')

    if not ytc.exists(args.user_ip_by_qid_table):
        parser.error('Table "{}" with mapping qid to user-ip do not exists'.format(args.user_ip_by_qid_table))

    logger.info('Start')
    logger.info('Source: %s', REDIR_LOG)
    logger.info('Destination: %s', args.destination)
    logger.info('Backup: %s', args.backup_destination)
    logger.info('Mapping gid: %s', args.user_ip_by_qid_table)
    logger.info('From: %s', args.from_date.strftime('%Y-%m-%d'))
    logger.info('To: %s', args.to_date.strftime('%Y-%m-%d'))

    with ytc.Transaction():
        ytc.lock(REDIR_LOG, waitable=True)
        ytc.create('map_node', args.destination, recursive=True)
        logger.info('%s created', args.destination)
        ytc.create('map_node', args.backup_destination, recursive=True)
        logger.info('%s created', args.backup_destination)

        user_ip_by_qid = {row['qid']: row['user_ip'] for row in ytc.read_table(args.user_ip_by_qid_table)}
        logger.info('Got %s qid mappings', len(user_ip_by_qid))

        for table in ytc.tables_for_daterange(REDIR_LOG, args.from_date, args.to_date):
            ytc.lock(table, waitable=True)

            destination_table = '{}/{}'.format(args.destination, table.split('/')[-1])
            backup_table = '{}/{}'.format(args.backup_destination, table.split('/')[-1])
            ytc.copy(table, backup_table)
            logger.info('Backup %s', backup_table)

            logger.info('From %s to %s', table, destination_table)
            ytc.create('table', destination_table, attributes={
                'schema': ytc.get_attribute(table, 'schema'),
                'processed': ytc.get_attribute(table, 'processed'),
                'compression_codec': ytc.get_attribute(table, 'compression_codec'),
                'optimize_for': ytc.get_attribute(table, 'optimize_for'),
            })

            logger.info('Processing %s (%d) ...', table, ytc.row_count(table))
            rows = []
            count = 0
            for row in ytc.read_table(table):
                if row['qid'] in user_ip_by_qid:
                    real_user_ip = user_ip_by_qid[row['qid']]
                    if real_user_ip != row['userip']:
                        row['userip'] = real_user_ip
                        count += 1
                rows.append(row)

            ytc.write_table(destination_table, rows)
            logger.info('Saved %s. Updated %d/%d redirects', destination_table, count, len(rows))

    logger.info('Done')
