# -*- encoding: utf-8 -*-
from collections import defaultdict
from datetime import datetime, timedelta
from optparse import OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger

ROOT_PATH = '//home/rasp'
BALANCE_LOG = '%s/logs/avia-redir-balance-log/latest' % ROOT_PATH
BALANCE_BY_DAY_ROOT = '%s/logs/avia-redir-balance-by-day-log' % ROOT_PATH
ALLOWED_ENVS = ['production', 'dev']


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from django.conf import settings

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser()

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-p', '--proxy', dest='proxy', default=settings.YT_PROXY)
    optparser.add_option('-d', '--days', dest='days', type='int', default=3)
    optparser.add_option('-l', '--left_date', dest='left_date')

    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)
    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    configure_wrapper(yt)
    if options.proxy != settings.YT_PROXY:
        yt.config['proxy']['url'] = options.proxy

    log.info('Start')

    current_env = settings.ENVIRONMENT
    if current_env not in ALLOWED_ENVS:
        allowed_envs_str = ', '.join(ALLOWED_ENVS)
        log.info('Current ENVIRONMENT %s. Only %s are allowed.' % (current_env, allowed_envs_str))
        return

    if options.left_date:
        left_date = datetime.strptime(options.left_date, '%Y-%m-%d').date()
    else:
        left_date = (datetime.now() - timedelta(days=options.days)).date()

    right_date = left_date + timedelta(days=options.days)

    log.info(
        'Read data from %s: %s - %s',
        BALANCE_LOG,
        left_date.strftime('%Y-%m-%d'),
        right_date.strftime('%Y-%m-%d'),
    )

    data = defaultdict(list)
    for record in yt.read_table(BALANCE_LOG, format=yt.JsonFormat(), raw=False):
        try:
            event_date = datetime.strptime(
                record.get('ISO_EVENTTIME', '')[:10], '%Y-%m-%d'
            ).date()
        except ValueError:
            log.error('Can\'t find/parse event time in %r', record)
            continue

        if not (left_date <= event_date <= right_date):
            continue

        data[event_date].append(record)

    if not yt.exists(BALANCE_BY_DAY_ROOT):
        log.info('Create %s', BALANCE_BY_DAY_ROOT)
        yt.create('map_node', BALANCE_BY_DAY_ROOT)

    for event_date, records in data.items():
        destination_table = '%s/%s' % (BALANCE_BY_DAY_ROOT,  event_date.strftime('%Y-%m-%d'))
        log.info('Write %s', destination_table)
        yt.write_table(destination_table, records, format=yt.JsonFormat())

    log.info('Done')
