# -*- encoding: utf-8 -*-
import json
import os
import sys
import time

from copy import copy
from datetime import date
from optparse import Option, OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger
from django.conf import settings


AURORA_ROOT = '//home/aurora/release/avia/aviatickets'
AVIA_ROOT = '//home/rasp'
ALLOWED_ENVS = ['production', 'dev']
TABLE_NODE_TYPE = 'table'

COMMON_FIELDS_SCHEMA = {
    '_read_unixtime': 'int64',
    '_source_table': 'string',
    'adults': 'int64',
    'aviacompany': 'string',
    'child': 'int64',
    'currency': 'string',
    'destination': 'string',
    'infant': 'int64',
    'one_way_ticket': 'boolean',
    'origin': 'string',
    'partner': 'string',
    'shift_date': 'string',
    'shift_days_before_flights': 'int64',
    'shift_week': 'string',
    'eventtime': 'int64',
    '_rest': 'any',
}

AVIATICKETS_LOG_SCHEMA = dict(
    minimal_price='double',
    time_travel='string',
    time_arrival='string',
    time_departure='string',
    **COMMON_FIELDS_SCHEMA
)

AVIATICKETS_STATS_LOG_SCHEMA = dict(
    one_stop_minimal_price='int64',
    direct_minimal_price='int64',
    two_plus_stop_minimal='int64',
    carrier='any',
    **COMMON_FIELDS_SCHEMA
)
AVIATICKETS_STATS_LOG_IGNORED_FILEDS = ['minimal_price', 'time_arrival', 'time_departure', 'time_travel', 'type']

AVIATICKETS_STATS_LOG_SOURCE_TABLES = {
    os.path.join(AURORA_ROOT, t)
    for t in ['skyscanner_ru', 'avia_yandex_ru', 'aviasales_ru']
}


def main():
    import travel.avia.admin.init_project  # noqa

    import logging

    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option("-p", "--proxy", dest="proxy", default=settings.YT_PROXY)

    options, args = optparser.parse_args()

    configure_wrapper(yt)
    if options.proxy != settings.YT_PROXY:
        yt.config['proxy']['url'] = options.proxy

    if options.verbose:
        add_stdout_handler(log)
    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    check_environment(log)
    sync(log)


def check_environment(log):
    current_env = settings.ENVIRONMENT
    if current_env not in ALLOWED_ENVS:
        allowed_envs_str = ', '.join(ALLOWED_ENVS)
        log.info('Current ENVIRONMENT %s. Run only %s allowed.' % (current_env, allowed_envs_str))
        sys.exit()


def sync(log):
    try:
        source_nodes = yt.list(AURORA_ROOT, absolute=True, attributes=['type'])
        unixtime = int(time.time())
        today = date.today().strftime('%Y-%m-%d')
        aviatickets_log = os.path.join(AVIA_ROOT, 'logs/aurora_aviatickets-log', today)
        aviatickets_stats_log = os.path.join(AVIA_ROOT, 'logs/aurora_aviatickets_stats-log', today)

        yt.mkdir(os.path.dirname(aviatickets_log), recursive=True)
        yt.mkdir(os.path.dirname(aviatickets_stats_log), recursive=True)
        create_table(aviatickets_log, AVIATICKETS_LOG_SCHEMA)
        create_table(aviatickets_stats_log, AVIATICKETS_STATS_LOG_SCHEMA)
        aviatickets_log_rows = []
        aviatickets_stats_log_rows = []

        for node in source_nodes:
            if node.attributes['type'] != TABLE_NODE_TYPE:
                continue
            log.info('Read data from: %s', node)
            for row in yt.read_table(node, format=yt.JsonFormat(encode_utf8=True, encoding='utf-8'), raw=False):
                if not row.get('value'):
                    log.info('Can\'t find value column in table %s', node)
                    break

                value = json.loads(row.get('value'))
                direction = value.pop('direction', None)

                if not direction:
                    log.info('Can\'t find direction in row')
                    continue

                common_row = build_common_row(value, unixtime, node, direction, log)
                aviatickets_log_rows.append(build_aviatickets_row(common_row, value, log))

                if str(node) in AVIATICKETS_STATS_LOG_SOURCE_TABLES:
                    aviatickets_stats_log_rows.append(build_aviatickets_stats_row(common_row, value, log))

        write_rows(aviatickets_log, aviatickets_log_rows, log)
        write_rows(aviatickets_stats_log, aviatickets_stats_log_rows, log)
        log.info('Done')
    except Exception:
        log.exception('ERROR')


def build_common_row(row, unixtime, source_table, direction, log):
    new_row = {
        '_source_table': os.path.basename(source_table),
        '_read_unixtime': unixtime,
        'destination': direction.get('destination'),
        'origin': direction.get('origin'),
    }
    new_row.update(row)

    safe_cast(new_row, '_read_unixtime', int, log)
    safe_cast(new_row, 'adults', int, log)
    safe_cast(new_row, 'child', int, log)
    safe_cast(new_row, 'infant', int, log)
    safe_cast(new_row, 'shift_days_before_flights', int, log)
    safe_cast(new_row, 'eventtime', int, log)
    safe_cast(new_row, 'one_way_ticket', boolean, log)

    return new_row


def build_aviatickets_row(common_row, row, log):
    result = row.copy()
    result.update(common_row)

    safe_cast(result, 'minimal_price', float, log)

    add_rest_field(result, AVIATICKETS_LOG_SCHEMA)
    return result


def build_aviatickets_stats_row(common_row, row, log):
    result = row.copy()
    result.update(common_row)

    if 'stats' in result:
        stats = result.pop('stats')
        fill_stats(result, stats, log)

    for field in AVIATICKETS_STATS_LOG_IGNORED_FILEDS:
        result.pop(field, None)

    add_rest_field(result, AVIATICKETS_STATS_LOG_SCHEMA)
    return result


def fill_stats(row, stats, log):
    if not stats:
        return

    row['direct_minimal_price'] = stats.get('direct_minimal_price')
    row['one_stop_minimal_price'] = stats.get('one_stop_minimal_price')
    row['two_plus_stop_minimal'] = stats.get('two_plus_stop_minimal')
    safe_cast(row, 'direct_minimal_price', int, log)
    safe_cast(row, 'one_stop_minimal_price', int, log)
    safe_cast(row, 'two_plus_stop_minimal', int, log)

    if stats.get('carriers'):
        row['carrier'] = {}
        for item in stats['carriers']:
            row['carrier'][item['carrier']] = item['carrier_minimal_price']


def create_table(table_name, schema):
    if not yt.exists(table_name):
        yt.create_table(
            table_name,
            attributes={
                'schema': [
                    {'name': key, 'type': value}
                    for key, value in schema.items()
                ],
                'optimize_for': 'scan',
            }
        )


def write_rows(table, rows, log):
    log.info('Save data to: %s', table)
    yt.write_table(table, rows, format=yt.JsonFormat(encode_utf8=True, encoding='utf-8'))


def boolean(value):
    if isinstance(value, str):
        value = value.lower().strip()
        if 'true' == value:
            return True
        elif 'false' == value:
            return False
        else:
            raise ValueError('Cannot convert %s to bool' % value)
    else:
        return bool(value)


def add_rest_field(dct, schema):
    # type: (dict, dict) -> None
    unknown_fields = set(dct.keys()) - set(schema.keys())
    dct['_rest'] = {field: dct[field] for field in unknown_fields}
    for field in unknown_fields:
        dct.pop(field)


def safe_cast(dct, key, target_type, logger, ignore_absent=True, default=None):
    """
    Приводим значение в словаре к определенному типу и записываем обратно в словарь.

    :param dict[str, any] dct: словарь, в котором нужно привести значение к определенному типу
    :param str key: ключ в словаре, в котором делается приведение
    :param target_type: тип, к которому нужно привести
    :param logging.Logger logger: логгер для ошибок
    :param bool ignore_absent: не выбрасывать исключение, если нет ключа в словаре
    :param default: значение по умолчанию, если тип привести нельзя
    """
    try:
        if key in dct:
            dct[key] = target_type(dct[key])
        else:
            logger.debug('Dict: %s. KeyError: %s', dct, key)
            if not ignore_absent:
                raise KeyError(key)
    except (ValueError, TypeError):
        logger.error(
            'Dict: %s. Cannot cast key %s value %s to %s. Replacing with default %s',
            dct,
            key,
            dct[key],
            str(target_type),
            default,
        )
        dct[key] = default
    return dct


class Yoption(Option):
    TYPES = Option.TYPES
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
