# -*- encoding: utf-8 -*-
import calendar
import json
import logging
import os

from copy import copy
from datetime import datetime, date
from optparse import Option, OptionParser

import yt.wrapper as yt
import yt.logger_config as yt_logger_config
import yt.logger as yt_logger


AURORA_ROOT = '//home/extdata/aurora/airport/timetable'
AVIA_ROOT = '//home/rasp'


def main():
    import travel.avia.admin.init_project  # noqa

    from dateutil.parser import parse
    from django.conf import settings
    from django.db import connection, transaction
    from collections import defaultdict

    from travel.avia.library.python.avia_data.models.tablo import Tablo
    from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
    from travel.avia.admin.lib.yt_helpers import configure_wrapper

    @transaction.atomic()
    def update_tablo_db(tablo_db_records):
        log.info('Clean MySQL table')
        cursor = connection.cursor()
        cursor.execute('DELETE FROM avia_tablo; ALTER TABLE avia_tablo AUTO_INCREMENT = 1;')
        cursor.close()

        log.info('Try to save %s records', len(tablo_db_records))
        Tablo.objects.bulk_create(tablo_db_records, batch_size=1000)

        tablo_count = Tablo.objects.all().count()
        log.info('%s record saved', tablo_count)

    def parse_date(datetime_str):
        if datetime_str:
            try:
                return parse(datetime_str)
            except Exception as e:
                log.warning('Can\'t parse date/time "%s": %s', datetime_str, e.message)
                return None

    bad_dates = defaultdict(set)

    log = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser(option_class=Yoption)

    optparser.add_option('-v', '--verbose', action='store_true')
    optparser.add_option('-p', '--proxy', dest='proxy', default=settings.YT_PROXY)

    options, args = optparser.parse_args()

    configure_wrapper(yt)
    if options.proxy != settings.YT_PROXY:
        yt.config['proxy']['url'] = options.proxy

    if options.verbose:
        add_stdout_handler(log)

    else:
        yt_logger_config.LOG_LEVEL = 'WARNING'
        reload(yt_logger)

    log.info('Start')

    try:
        source_tables = yt.list(AURORA_ROOT, absolute=True)
        destination_lines = []
        unixtime = calendar.timegm(datetime.utcnow().utctimetuple())
        destination_table_name = os.path.join(
            AVIA_ROOT,
            'logs/aurora-timetable-log',
            date.today().strftime('%Y-%m-%d')
        )
        destination_table = yt.TablePath(destination_table_name, append=True)

        yt.mkdir(os.path.dirname(destination_table_name), recursive=True)

        tablo_db_records = []

        for t in source_tables:
            log.info('Read data from: %s', t)
            for row in yt.read_table(t, format=yt.JsonFormat(encode_utf8=True, encoding='utf-8'), raw=False):
                value = row.get('value')
                try:
                    if isinstance(value, (unicode, str)):
                        value = json.loads(value)

                except ValueError as e:
                    log.exception('Can\'t parse %s: %s', t, e)
                    continue

                if not isinstance(value, dict):
                    log.error('Bad data in %s: %r', t, row.get('value'))
                    continue

                # Сделаем каждый рейс в одной строке
                flights = value.pop('flights')

                for flight in flights:
                    record = {
                        '_source_table': t.split('/')[-1],
                        '_read_unixtime': unixtime,
                        'gates': None,  # Не для всех аэропортов парсится "выход", но мы хотим эти столбцы в YT
                        'terminal': None,  # Аналогично столбцу "выход"
                    }

                    record.update(value)

                    try:
                        record.update(flight)

                    except ValueError:
                        log.warning('Bad flight in: %s (%r)', t, flight)
                        continue

                    destination_lines.append(record)

                    # Обновим табло
                    flight_number = record.get('flight_number') or record.get('airline_number')
                    airport_iata = record.get('airport')
                    airline_iata = record.get('airline_code')
                    unparsed_date = record.get('date')
                    unparsed_time_scheduled = record.get('time_scheduled')
                    unparsed_time_actual = record.get('time_actual')
                    date_scheduled = parse_date(unparsed_date)
                    time_scheduled = parse_date(unparsed_time_scheduled)
                    time_actual = parse_date(unparsed_time_actual)
                    destination = record.get('destination')

                    if len(flight_number) > 10:
                        log.warning(u'Wrong "flight_number": %s', flight_number)
                        continue

                    if len(airport_iata) > 3:
                        log.warning(u'Wrong "airport_iata": %s', airport_iata)
                        continue

                    if len(airline_iata) > 3:
                        log.warning(u'Wrong "airline_iata": %s', airline_iata)
                        continue

                    if not destination:
                        log.warning(u'Empty "destination": %s', record)
                        continue

                    if not date_scheduled and unparsed_date:
                        bad_dates[airport_iata].add(unparsed_date)

                    if not time_scheduled and unparsed_time_scheduled:
                        bad_dates[airport_iata].add(unparsed_time_scheduled)

                    if not time_actual and unparsed_time_actual:
                        bad_dates[airport_iata].add(unparsed_time_actual)

                    tablo_record = Tablo(
                        airport_iata=airport_iata,
                        event_type=record.get('type'),
                        flight_number=flight_number,
                        airline_iata=airline_iata,
                        destination=destination,
                        date_scheduled=date_scheduled,
                        time_scheduled=time_scheduled,
                        time_actual=time_actual,
                        status=record.get('status') or ''
                    )

                    tablo_db_records.append(tablo_record)

        try:
            update_tablo_db(tablo_db_records)
        except Exception:
            log.exception('Error while update database')

        current_env = settings.ENVIRONMENT
        if current_env == 'production':
            log.info('Save %s lines to: %s', len(destination_lines), destination_table)
            yt.write_table(
                destination_table, destination_lines,
                format=yt.JsonFormat(encode_utf8=True, encoding='utf-8')
            )

        # Залогируем плохие даты
        for airport_iata, bad_items in bad_dates.items():
            message = u'Bad dates {airport_iata}: {bad_items}'.format(
                airport_iata=airport_iata,
                bad_items=u', '.join(bad_items)
            )
            log.info(message)

        log.info('Done')

    except Exception:
        log.exception('ERROR')


class Yoption(Option):
    TYPES = Option.TYPES
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
