# -*- encoding: utf-8 -*-
import travel.avia.admin.init_project  # noqa

import logging
from copy import copy
from optparse import Option, OptionParser

from django.db import connection, transaction

from travel.avia.library.python.common.models.geo import StationCode
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import iter_protobuf_data, ResourceType

SIRENA_CODESYSTEM_ID = 1
IATA_CODESYSTEM_ID = 4
ICAO_CODESYSTEM_ID = 5


def main():
    log = logging.getLogger(__name__)
    create_current_file_run_log()

    optparser = OptionParser(option_class=Yoption)
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('Start')

    try:
        station_data = iter_protobuf_data(ResourceType.TRAVEL_DICT_RASP_STATION_PROD)
        sync_stations(station_data, log)

    except Exception as e:
        log.exception('ERROR: %s', e.message)


@transaction.atomic()
def sync_stations(stations_data, log):
    stations_data = list(stations_data)
    log.info('Start updating ids')
    cursor = connection.cursor()

    unmatched_stations_with_codes = StationCode.objects.filter(
        system_id__in=[SIRENA_CODESYSTEM_ID, IATA_CODESYSTEM_ID, ICAO_CODESYSTEM_ID]
    ).values('station_id')
    unmatched_stations_with_codes = set(x['station_id'] for x in unmatched_stations_with_codes)

    log.info('%d stations with codes are found in database', len(unmatched_stations_with_codes))
    for index, station in enumerate(stations_data):
        match = sync_station_id(station, cursor, log)
        if match:
            if match not in unmatched_stations_with_codes:
                log.warn('Station %s is matched again', match)
            else:
                unmatched_stations_with_codes.remove(match)
        if (index + 1) % 100 == 0:
            log.info('Processed %d entries', index + 1)
    cursor.close()
    log.info('Done updating ids. %s protobuf entries processed', len(stations_data))


def sync_station_id(station, cursor, log):
    if not station.StationCodes:
        return
    for system_id in [IATA_CODESYSTEM_ID, SIRENA_CODESYSTEM_ID, ICAO_CODESYSTEM_ID]:
        if system_id in station.StationCodes:
            station_ids = StationCode.objects\
                .filter(system_id=system_id, code=station.StationCodes[system_id])\
                .values('station_id')
            if station_ids:
                old_id = station_ids[0]['station_id']
                if old_id != station.Id:
                    log.info('updating: new id %s, old id %s', station.Id, old_id)
                    cursor.execute('UPDATE www_station SET id = %s WHERE id = %s',
                                   [station.Id, old_id])
                return old_id


class Yoption(Option):
    TYPES = Option.TYPES
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
