# coding: utf-8
from __future__ import unicode_literals, print_function, division

"""
Скрипт запускает синхронизацию таблиц базы данных с расписаниями в правильном порядке.

По-умолчанию запускаются все скрипты.
Можно задать набор скриптов синхронизации, которые нужно пропустить,
через аргумент коммандной строки --exclude, например, --exclude=station,region
"""
import travel.avia.admin.init_project  # noqa

import argparse
import logging
from collections import namedtuple
from enum import Enum

from django.db import transaction

from travel.avia.admin.avia_scripts.sync_with_rasp import (
    sync_www_company_with_rasp, sync_www_country, sync_www_region, sync_www_station_with_rasp,
    sync_currency_currency, sync_www_settlement, sync_www_aviaalliance, sync_www_pointsynonym,
)
from travel.avia.library.python.common.models.geo import CodeSystem
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log

log = logging.getLogger(__name__)


class ScriptType(Enum):
    AVIAALLIANCE = 'aviaalliance'
    COMPANY = 'company'
    COUNTRY = 'country'
    CURRENCY = 'currency'
    POINTSYNONYM = 'pointsynonym'
    REGION = 'region'
    SETTLEMENT = 'settlement'
    STATION = 'station'


ScriptInfo = namedtuple('ScriptInfo', 'type, callable')

# Скрипты должны располагаться в правильном порядке, в этом порядке они и будут вызываться.
SCRIPTS = [
    ScriptInfo(ScriptType.AVIAALLIANCE, sync_www_aviaalliance.do_sync),
    ScriptInfo(ScriptType.CURRENCY, sync_currency_currency.do_sync),
    ScriptInfo(ScriptType.COUNTRY, sync_www_country.do_sync),
    ScriptInfo(ScriptType.REGION, sync_www_region.do_sync),
    ScriptInfo(ScriptType.SETTLEMENT, sync_www_settlement.do_sync),
    ScriptInfo(ScriptType.STATION, sync_www_station_with_rasp.do_sync),
    ScriptInfo(ScriptType.COMPANY, sync_www_company_with_rasp.do_sync),
    ScriptInfo(ScriptType.POINTSYNONYM, sync_www_pointsynonym.do_sync),
]


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    parser.add_argument('--exclude', help='Comma separated excluded scripts list.')
    parser.add_argument('--ignore-unknown-exclude', action='store_true',
                        help='Ignore wrong exclude. May be useful to set exclude before deploy of new sync script.')
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(log)
    create_current_file_run_log()

    log.info('Start precache...')
    TransportType.objects.precache()
    CodeSystem.objects.precache()
    log.info('Done precache')

    log.info('Start sync all')
    disabled_script_types = _get_disabled_script_types(args.exclude, args.ignore_unknown_exclude)
    try:
        do_sync(disabled_script_types)
    except Exception:
        log.exception('Error in sync all')
        raise
    else:
        log.info('Successfully sync all')


@transaction.atomic
def do_sync(disabled_script_types):
    for script in SCRIPTS:
        if script.type in disabled_script_types:
            log.info('=== Skip sync %s ===', script.type.value)
        else:
            log.info('=== Start sync %s ===', script.type.value)
            script.callable()


def _get_disabled_script_types(exclude, ignore_unknown_exclude=False):
    """
    Получаем скрипты синхронизации, которые нужно пропустить.

    Если при разборе упадем, то это ок, потому что пользователь явно увидит, что скрипты заданы неправильно.
    """
    disabled_scripts = (exclude or '').strip()
    disabled_script_types = []
    for s in disabled_scripts.split(','):
        s = s.strip()
        if not s:
            continue
        try:
            disabled_script_types.append(ScriptType(s.lower()))
        except Exception:
            if ignore_unknown_exclude:
                log.warning('Unknown script type "%s"', s)
            else:
                raise
    return disabled_script_types
