# coding: utf-8
from __future__ import unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import argparse
import logging
from collections import namedtuple

import six
from django.conf import settings
from django.db import transaction
from django.utils.encoding import force_text

from travel.avia.library.python.common.models.currency import Currency
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType

log = logging.getLogger(__name__)

MIN_NOT_RASP_CURRENCY_ID = 10000
CurrencyMapping = namedtuple('CurrencyMapping', 'rasp_currency, currency')


@transaction.atomic
def sync_currency(currency_repository):
    def get_currency_key(currency):
        return currency.code

    def get_rasp_currency_key(rasp_currency):
        return rasp_currency.Code

    currency_by_ids = {c.id: c for c in Currency.objects.all()}
    currency_by_keys = {get_currency_key(c): c for c in six.itervalues(currency_by_ids)}

    mappings = []
    for rasp_currency in currency_repository.itervalues():
        currency = currency_by_ids.get(rasp_currency.Id) or currency_by_keys.get(get_rasp_currency_key(rasp_currency))
        mappings.append(CurrencyMapping(rasp_currency, currency))

    currency_ids_in_rasp = {m.currency.id for m in mappings if m.currency}
    currency_not_in_rasp = [c for c_id, c in six.iteritems(currency_by_ids) if c_id not in currency_ids_in_rasp]

    _shift_and_hide_currency(currency_not_in_rasp, max_id=max(currency_by_ids))
    _process_mapped_currency(mappings)


def _shift_and_hide_currency(currency_not_in_rasp, max_id):
    new_id = max(max_id + 1, MIN_NOT_RASP_CURRENCY_ID)
    for currency in currency_not_in_rasp:
        currency_id = currency.id
        if currency.id < MIN_NOT_RASP_CURRENCY_ID:
            currency_id = new_id
            Currency.objects.filter(id=currency.id).update(id=currency_id)
            new_id += 1
        log.info('Not in rasp %s', _currency_to_text(
            currency if currency_id == currency.id else Currency.objects.get(id=currency_id)
        ))


def _process_mapped_currency(mappings):
    for mapping in mappings:
        rasp_currency = mapping.rasp_currency
        currency = mapping.currency
        spec = {
            'id': rasp_currency.Id,
            'code': rasp_currency.Code,
            'iso_code': rasp_currency.IsoCode,

            'name': rasp_currency.Title.Ru,
            'name_in': rasp_currency.TitleIn.Ru,
            'name_tr': rasp_currency.Title.Tr,
            'name_uk': rasp_currency.Title.Uk,
            'name_uk_in': rasp_currency.TitleIn.Uk,

            'template': rasp_currency.Template.Ru,
            'template_whole': rasp_currency.TemplateWhole.Ru,
            'template_cents': rasp_currency.TemplateCents.Ru,
            'template_tr': rasp_currency.Template.Tr,
            'template_whole_tr': rasp_currency.TemplateWhole.Tr,
            'template_cents_tr': rasp_currency.TemplateCents.Tr,

            'order': rasp_currency.Order,
            'order_tr': rasp_currency.OrderTr,
            'order_ua': rasp_currency.OrderUa,
        }
        if not currency:
            currency = Currency.objects.create(**spec)
            log.info('Created %s', _currency_to_text(currency))
        else:
            Currency.objects.filter(id=currency.id).update(**spec)


def _currency_to_text(currency):
    return 'Currency id={id}, code={code}, name="{name}"'.format(
        id=currency.id,
        code=currency.code,
        name=force_text(currency.name),
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(log)
    create_current_file_run_log()

    log.info('Start sync currency_currency')

    try:
        do_sync()
    except Exception:
        log.exception('Error in sync currency_currency')
        raise
    else:
        log.info('Successfully sync currency_currency')


def do_sync():
    currency_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_CURRENCY_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    sync_currency(currency_repository)
