# -*- encoding: utf-8 -*-
from __future__ import unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import logging
from copy import copy
from optparse import Option, OptionParser

import six
from django.conf import settings
from django.db import transaction
from django.db.models import Q
from django_bulk_update.helper import bulk_update

from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.geo import Station
from travel.avia.admin.lib.iterators import chunker
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType

log = logging.getLogger(__name__)

COMPANY_UPDATE_FIELDS = (
    'address',
    'home_station_id',
    'supplier_code',
    'sirena_id',
    'iata',
    'icao',
    'icao_ru',
    't_type_id',
    'is_freight',
    'priority',
    'email',
    'contact_info',
    'phone',
    'phone_booking',
    'description',
    'logo',
    'icon',
    'logo_mono',
    'country_id',
    'hidden',
    'strange',
    'meta_title',
    'meta_description',
    'alliance_id',
    'url',
    'title',
    'title_en',
    'title_ru',
    'title_uk',
    'title_tr',
    'short_title',
    'short_title_ru',
    'short_title_en',
    'short_title_uk',
    'short_title_tr',
    'bonus_name',
    'bonus_name_ru',
    'bonus_name_en',
    'bonus_name_uk',
    'bonus_name_tr',
    'registration_phone',
    'registration_phone_ru',
    'registration_phone_en',
    'registration_phone_uk',
    'registration_phone_tr',
    'registration_url',
    'registration_url_ru',
    'registration_url_en',
    'registration_url_uk',
    'registration_url_tr',
)

COMPANY_NEW_TITLE_UPDATE_FIELDS = (
    'ru_nominative',
    'en_nominative',
    'uk_nominative',
    'tr_nominative',
)


def main():
    create_current_file_run_log()

    optparser = OptionParser(option_class=Yoption)
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('Start')

    try:
        do_sync()
    except Exception:
        log.exception('Error in sync company.')
        raise


def do_sync():
    company_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_CARRIER_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    sync_companies(company_repository)


@transaction.atomic()
def sync_companies(company_repository):
    companies_data = list(company_repository.itervalues())
    _zero_out_deleted_companies(companies_data)
    _clear_uniq_fields_that_are_in_rasp(companies_data)
    _update_companies(companies_data)


def _zero_out_deleted_companies(companies_data):
    ids_in_database = set(Company.objects.values_list('pk', flat=True))
    log.info('%d companies in database', len(ids_in_database))
    for datum in companies_data:
        if datum.Id in ids_in_database:
            ids_in_database.remove(datum.Id)

    updates_for_deleted = {'hidden': True, 'icao': None, 'iata': None, 'sirena_id': None, 'icao_ru': None, 'slug': None,
                           'supplier_code': None}
    log.info(
        '%d companies are present in database, but were not found in protobuf. They will be updated with following '
        'values: %s',
        len(ids_in_database),
        updates_for_deleted
    )

    log.info('Current values')
    fields_to_log = list(updates_for_deleted) + ['id']
    deleted_companies = Company.objects.filter(pk__in=ids_in_database)
    for company in deleted_companies:
        log.info(str({k: getattr(company, k) for k in fields_to_log}))
    deleted_companies.update(**updates_for_deleted)
    log.info('Deleted companies update completed')


def _clear_uniq_fields_that_are_in_rasp(companies_data):
    for field, proto_attr in (
        ('supplier_code', 'SupplierCode'),
        ('sirena_id', 'SirenaId'),
        ('icao', 'Icao'),
        ('icao_ru', 'IcaoRu'),
    ):
        for chunk in chunker(companies_data, 1000):
            filter_condition = Q()
            for rasp_company in chunk:
                filter_condition |= Q(**{field: getattr(rasp_company, proto_attr)})
            Company.objects.filter(filter_condition).update(**{field: None})


def _update_companies(companies_data):
    log.info('%s companies in protobuf', len(companies_data))
    companies = {
        company.id: company
        for company in Company.objects.all()
    }

    companies_to_update = []
    company_new_titles_to_update = []
    home_station_ids = _get_home_station_ids(companies_data)
    for rasp_company in companies_data:
        if rasp_company.HomeStationId in home_station_ids:
            home_station_id = rasp_company.HomeStationId
        else:
            home_station_id = None
            if rasp_company.HomeStationId:
                log.warning('Station with id %s not found', rasp_company.HomeStationId)

        spec = {
            'id': rasp_company.Id,
            'address': rasp_company.Address,
            'home_station_id': home_station_id,
            'supplier_code': rasp_company.SupplierCode or None,
            'sirena_id': rasp_company.SirenaId or None,
            'iata': rasp_company.Iata,
            'icao': rasp_company.Icao or None,
            'icao_ru': rasp_company.IcaoRu or None,
            't_type_id': rasp_company.TransportType or None,
            'is_freight': rasp_company.IsFreight,
            'priority': rasp_company.Priority,
            'email': rasp_company.Email,
            'contact_info': rasp_company.ContactInfo,
            'phone': rasp_company.Phone,
            'phone_booking': rasp_company.PhoneBooking,
            'description': rasp_company.Description,
            'logo': rasp_company.Logo,
            'icon': rasp_company.Icon,
            'logo_mono': rasp_company.LogoMono,
            'country_id': rasp_company.CountryId or None,
            'hidden': rasp_company.IsHidden,
            'strange': rasp_company.IsStrange,
            'meta_title': rasp_company.MetaTitle,
            'meta_description': rasp_company.MetaDescription,
            'alliance_id': rasp_company.AllianceId or None,
            'url': rasp_company.Url,
            'title': rasp_company.Title,
            'title_en': rasp_company.TitleEn,
            'title_ru': rasp_company.TitleRu,
            'title_uk': rasp_company.TitleUk,
            'title_tr': rasp_company.TitleTr,
            'short_title': rasp_company.ShortTitle,
            'short_title_ru': rasp_company.ShortTitleRu,
            'short_title_en': rasp_company.ShortTitleEn,
            'short_title_uk': rasp_company.ShortTitleUk,
            'short_title_tr': rasp_company.ShortTitleTr,
            'bonus_name': rasp_company.BonusName,
            'bonus_name_ru': rasp_company.BonusNameRu,
            'bonus_name_en': rasp_company.BonusNameEn,
            'bonus_name_uk': rasp_company.BonusNameUk,
            'bonus_name_tr': rasp_company.BonusNameTr,
            'registration_phone': rasp_company.RegistrationPhone,
            'registration_phone_ru': rasp_company.RegistrationPhoneRu,
            'registration_phone_en': rasp_company.RegistrationPhoneEn,
            'registration_phone_uk': rasp_company.RegistrationPhoneUk,
            'registration_phone_tr': rasp_company.RegistrationPhoneTr,
            'registration_url': rasp_company.RegistrationUrl,
            'registration_url_ru': rasp_company.RegistrationUrlRu,
            'registration_url_en': rasp_company.RegistrationUrlEn,
            'registration_url_uk': rasp_company.RegistrationUrlUk,
            'registration_url_tr': rasp_company.RegistrationUrlTr,
        }
        if rasp_company.Id in companies:
            company = companies[rasp_company.Id]
            for field, value in six.iteritems(spec):
                setattr(company, field, value)
            companies_to_update.append(company)
        else:
            company = Company.objects.create(**spec)

        title = company.new_L_title
        title.ru_nominative = rasp_company.TitleRu or None
        title.en_nominative = rasp_company.TitleEn or None
        title.uk_nominative = rasp_company.TitleUk or None
        title.tr_nominative = rasp_company.TitleTr or None

        company_new_titles_to_update.append(title)

    log.info("companies for update count: {}".format(len(companies_to_update)))

    bulk_update(companies_to_update, batch_size=1000, update_fields=COMPANY_UPDATE_FIELDS)
    bulk_update(company_new_titles_to_update, batch_size=1000, update_fields=COMPANY_NEW_TITLE_UPDATE_FIELDS)


def _get_home_station_ids(companies_data):
    station_ids = [rasp_company.HomeStationId for rasp_company in companies_data if rasp_company.HomeStationId]
    return set(Station.objects.filter(id__in=station_ids).values_list('id', flat=True))


class Yoption(Option):
    TYPES = Option.TYPES
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
