# coding: utf-8
from __future__ import unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import argparse
import logging

import six
from django.conf import settings
from django.utils.encoding import force_text
from django_bulk_update.helper import bulk_update

from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Country
from travel.avia.admin.lib.iterators import chunker
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType

log = logging.getLogger(__name__)

COUNTRY_UNIQUE_FIELDS_AND_PROTO_ATTRS = (
    ('_geo_id', 'GeoId'),
    ('code', 'Code'),
    ('code3', 'Code3'),
    ('domain_zone', 'DomainZone'),
)

COUNTRY_UPDATE_FIELDS = (
    '_geo_id',
    'code',
    'title',
    'code3',
    '_kladr_id',
    'domain_zone',
    'currency_id',
    'language',
    'title_ru',
    'title_ru_genitive',
    'title_ru_locative',
    'title_ru_preposition_v_vo_na',
    'title_en',
    'title_uk',
    'title_uk_accusative',
    'title_tr',
)

COUNTRY_NEW_TITLE_UPDATE_FIELDS = (
    'ru_nominative',
    'ru_genitive',
    'ru_locative',
    'en_nominative',
    'uk_nominative',
    'uk_accusative',
    'tr_nominative',
)


def sync_country(rasp_country_repository):
    rasp_country_data = list(rasp_country_repository.itervalues())

    _log_deleted_countries(rasp_country_data)
    _clear_unique_fields_that_are_in_rasp(rasp_country_data)
    _update_countries(rasp_country_data)


def _log_deleted_countries(rasp_country_data):
    db_ids = set(Country.objects.all().values_list('id', flat=True))
    rasp_ids = {c.Id for c in rasp_country_data}
    deleted_ids = db_ids - rasp_ids
    if not deleted_ids:
        log.info('No deleted countries')
    else:
        for country in Country.objects.filter(id__in=deleted_ids):
            log.warning('%s was deleted in rasp', _country_to_text(country))


def _update_countries(rasp_country_data):
    countries_to_update = []
    country_new_titles_to_update = []

    country_by_ids = {country.id: country for country in Country.objects.all()}
    currency_ids = set(Currency.objects.all().values_list('id', flat=True))
    for rasp_country in rasp_country_data:
        currency_id = rasp_country.CurrencyId or None
        if currency_id and (currency_id not in currency_ids):
            log.warning('Unknown CurrencyId=%s in rasp proto', currency_id)
            currency_id = None

        spec = {
            'id': rasp_country.Id,
            '_geo_id': rasp_country.GeoId or None,
            'code': rasp_country.Code,
            'title': rasp_country.TitleDefault,
            'code3': rasp_country.Code3,
            '_kladr_id': rasp_country.KladrId,
            'domain_zone': rasp_country.DomainZone or None,
            'currency_id': currency_id,
            'language': rasp_country.Language,
            'title_ru': rasp_country.Title.Ru.Nominative,
            'title_ru_genitive': rasp_country.Title.Ru.Genitive,
            'title_ru_locative': rasp_country.Title.Ru.Prepositional,
            'title_ru_preposition_v_vo_na': rasp_country.Title.Ru.LocativePreposition,
            'title_en': rasp_country.Title.En.Nominative,
            'title_uk': rasp_country.Title.Uk.Nominative,
            'title_uk_accusative': rasp_country.Title.Uk.Accusative,
            'title_tr': rasp_country.Title.Tr.Nominative,
        }
        if rasp_country.Id not in country_by_ids:
            country = Country.objects.create(**spec)
            log.info('New country %s', _country_to_text(country))
        else:
            country = country_by_ids[rasp_country.Id]
            for field, value in six.iteritems(spec):
                setattr(country, field, value)
            countries_to_update.append(country)

        country.new_L_title.ru_nominative = rasp_country.Title.Ru.Nominative
        country.new_L_title.ru_genitive = rasp_country.Title.Ru.Genitive
        country.new_L_title.ru_locative = rasp_country.Title.Ru.Prepositional
        country.new_L_title.en_nominative = rasp_country.Title.En.Nominative
        country.new_L_title.uk_nominative = rasp_country.Title.Uk.Nominative
        country.new_L_title.uk_accusative = rasp_country.Title.Uk.Accusative
        country.new_L_title.tr_nominative = rasp_country.Title.Tr.Nominative
        country_new_titles_to_update.append(country.new_L_title)

    bulk_update(countries_to_update, batch_size=1000, update_fields=COUNTRY_UPDATE_FIELDS)
    bulk_update(country_new_titles_to_update, batch_size=1000, update_fields=COUNTRY_NEW_TITLE_UPDATE_FIELDS)


def _clear_unique_fields_that_are_in_rasp(rasp_country_data):
    for field, proto_attr in COUNTRY_UNIQUE_FIELDS_AND_PROTO_ATTRS:
        field_values = {getattr(rasp_country, proto_attr) for rasp_country in rasp_country_data}
        for values in chunker(field_values, 1000):
            Country.objects.filter(**{'{}__in'.format(field): values}).update(**{field: None})


def _country_to_text(country):
    return 'Country id={id}, code="{code}", geo_id={geo_id}, title="{title}"'.format(
        id=country.id,
        code=country.code,
        geo_id=country._geo_id,
        title=force_text(country.title),
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(log)
    create_current_file_run_log()

    log.info('Start sync www_country')

    try:
        do_sync()
    except Exception:
        log.exception('Error in sync www_country')
        raise
    else:
        log.info('Successfully sync www_country')


def do_sync():
    rasp_country_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_COUNTRY_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    sync_country(rasp_country_repository)
