# coding: utf-8
from __future__ import unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import argparse
import logging
from itertools import count
from past.builtins import long

from django.conf import settings
from django_bulk_update.helper import bulk_update
from six import iteritems, ensure_text
from six.moves import zip

from travel.avia.library.python.common.models.geo import PointSynonym
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import iter_protobuf_data, ResourceType

log = logging.getLogger(__name__)

POINTSYNONYM_UNIQUE_FIELDS_AND_PROTO_ATTRS = (
    ('title', 'Title'),
    ('content_type_id', 'ContentTypeId'),
    ('object_id', 'ObjectId'),
)

POINTSYNONYM_UPDATE_FIELDS = ('search_type', 'language')


def sync_pointsynonym(rasp_pointsynonym_data):
    _update_pointsynonym(rasp_pointsynonym_data)


def _update_pointsynonym(rasp_pointsynonym_data):
    db_objects = list(PointSynonym.objects.all())
    primary_key_conflicts = _pk_conflicts(rasp_pointsynonym_data, db_objects)
    max_id = max(ids_from_rasp_protos(rasp_pointsynonym_data) | ids_from_avia_models(db_objects))
    shift_pk_to_resolve_pk_conflicts(primary_key_conflicts, max_id)

    # read pointsynonyms from DB again, because some may change id
    avia_unique_dict = _avia_unique_dict(PointSynonym.objects.all())
    update_fields_from_protos(rasp_pointsynonym_data, avia_unique_dict)


def ids_from_rasp_protos(rasp_pointsynonym_data):
    return {synonym.Id for synonym in rasp_pointsynonym_data}


def ids_from_avia_models(avia_pointsynonyms):
    return {synonym.id for synonym in avia_pointsynonyms}


def _rasp_proto_by_unique(rasp_pointsynonym_data):
    return {unique_key_from_rasp_proto(rasp_proto): rasp_proto for rasp_proto in rasp_pointsynonym_data}


def _avia_unique_dict(avia_models):
    return {unique_key_from_avia_model(avia_model): avia_model for avia_model in avia_models}


def _pk_conflicts(rasp_protos, avia_models):
    rasp_unique_by_pk = {}
    rasp_pk_by_unique = {}
    for rasp_proto in rasp_protos:
        unique = unique_key_from_rasp_proto(rasp_proto)
        rasp_unique_by_pk[rasp_proto.Id] = unique
        rasp_pk_by_unique[unique] = rasp_proto.Id

    avia_pk_conflicts = set()
    for avia_model in avia_models:
        unique = unique_key_from_avia_model(avia_model)
        avia_id = avia_model.id
        if avia_id not in rasp_unique_by_pk:
            continue
        if unique == rasp_unique_by_pk[avia_id]:
            continue
        avia_pk_conflicts.add(avia_id)

    return avia_pk_conflicts


def unique_key_from_rasp_proto(rasp_pointsynonym):
    return ensure_text(rasp_pointsynonym.Title), long(rasp_pointsynonym.ContentTypeId), long(rasp_pointsynonym.ObjectId)


def unique_key_from_avia_model(avia_pointsynonym):
    return ensure_text(avia_pointsynonym.title), long(avia_pointsynonym.content_type_id), long(avia_pointsynonym.object_id)


def shift_pk_to_resolve_pk_conflicts(avia_pk_conflicts, max_id):
    for id_, new_id in zip(avia_pk_conflicts, count(start=max_id + 1)):
        PointSynonym.objects.filter(id=id_).update(id=new_id)
        log.info(
            'Conflicting point synonym moved from id = {id_} to {new_id}'.format(
                id_=id_,
                new_id=new_id,
            ),
        )
        new_id += 1


def update_fields_from_protos(rasp_pointsynonyms, avia_unique_dict):
    rasp_proto_by_unique = _rasp_proto_by_unique(rasp_pointsynonyms)

    pointsynonyms_to_update = []
    for unique, rasp_synonym in iteritems(rasp_proto_by_unique):
        if unique in avia_unique_dict:
            avia_pointsynonym = avia_unique_dict[unique]
            log.info('Update existing pointsynonym {}'.format(unique))
            if avia_pointsynonym.id == rasp_synonym.Id:
                avia_pointsynonym.search_type = rasp_synonym.SearchType
                avia_pointsynonym.language = rasp_synonym.Language
                pointsynonyms_to_update.append(avia_pointsynonym)
            else:
                PointSynonym.objects.filter(
                    title=rasp_synonym.Title,
                    content_type_id=rasp_synonym.ContentTypeId,
                    object_id=rasp_synonym.ObjectId,
                ).update(
                    id=rasp_synonym.Id,
                    search_type=rasp_synonym.SearchType,
                    language=rasp_synonym.Language,
                )
        else:
            conflicting_synonym = PointSynonym.objects.filter(
                title=rasp_synonym.Title,
                content_type_id=rasp_synonym.ContentTypeId,
                object_id=rasp_synonym.ObjectId,
            ).first()
            if conflicting_synonym is not None:
                log.info('Conflicting synonym found: {}. Updating {}'.format(
                    conflicting_synonym.id,
                    unique,
                ))
                PointSynonym.objects.filter(id=conflicting_synonym.id).update(
                    id=rasp_synonym.Id,
                    title=rasp_synonym.Title,
                    content_type_id=rasp_synonym.ContentTypeId,
                    object_id=rasp_synonym.ObjectId,
                    search_type=rasp_synonym.SearchType,
                    language=rasp_synonym.Language,
                )
            else:
                log.info('Create new pointsynonym {}'.format(unique))
                PointSynonym.objects.update_or_create(
                    title=rasp_synonym.Title,
                    content_type_id=rasp_synonym.ContentTypeId,
                    object_id=rasp_synonym.ObjectId,
                    defaults=dict(
                        id=rasp_synonym.Id,
                        search_type=rasp_synonym.SearchType,
                        language=rasp_synonym.Language,
                    )
                )

    bulk_update(pointsynonyms_to_update, batch_size=10000, update_fields=POINTSYNONYM_UPDATE_FIELDS)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(log)
    create_current_file_run_log()

    log.info('Start sync www_pointsynonym')

    try:
        do_sync()
    except Exception:
        log.exception('Error in sync www_pointsynonym')
        raise
    else:
        log.info('Successfully sync www_pointsynonym')


def do_sync():
    rasp_pointsynonym_data = list(iter_protobuf_data(
        ResourceType.TRAVEL_DICT_RASP_POINTSYNONYM_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    ))
    sync_pointsynonym(rasp_pointsynonym_data)
