# coding: utf-8
from __future__ import unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import argparse
import logging
from collections import namedtuple

import six
from django.conf import settings
from django.db import transaction
from django.utils.encoding import force_text
from django_bulk_update.helper import bulk_update

from travel.avia.library.python.common.models.geo import Region
from travel.avia.admin.lib.iterators import chunker
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType

log = logging.getLogger(__name__)

MIN_NOT_RASP_REGION_ID = 200000
REGION_UNIQUE_FIELDS_AND_PROTO_ATTRS = (
    ('_geo_id', 'GeoId'),
    ('koatuu', 'Koatuu'),
)
RegionMapping = namedtuple('RegionMapping', 'rasp_region, region')

REGION_UPDATE_FIELDS = (
    '_geo_id',
    'agent_geo_id',
    'title',
    '_kladr_id',
    'koatuu',
    'country_id',
    'time_zone',
    'hidden',
    'disputed_territory',
    'title_ru',
    'title_en',
    'title_uk',
    'title_tr',
)

REGION_NEW_TITLE_UPDATE_FIELDS = (
    'ru_nominative',
    'en_nominative',
    'uk_nominative',
    'tr_nominative',
)


@transaction.atomic
def sync_region(region_repository, tz_repository):
    def get_region_key(region):
        return '{geo_id}##{title}'.format(geo_id=region._geo_id, title=force_text(region.title))

    def get_rasp_region_key(rasp_region):
        return '{geo_id}##{title}'.format(geo_id=rasp_region.GeoId, title=force_text(rasp_region.TitleDefault))

    rasp_region_data = list(region_repository.itervalues())

    region_by_ids = {r.id: r for r in Region.objects.all()}
    region_by_keys = {get_region_key(r): r for r in six.itervalues(region_by_ids)}

    mappings = []
    for rasp_region in rasp_region_data:
        region = region_by_ids.get(rasp_region.Id) or region_by_keys.get(get_rasp_region_key(rasp_region))
        mappings.append(RegionMapping(rasp_region, region))

    region_ids_in_rasp = {m.region.id for m in mappings if m.region}
    regions_not_in_rasp = [r for r_id, r in six.iteritems(region_by_ids) if r_id not in region_ids_in_rasp]

    _shift_and_hide_regions(regions_not_in_rasp, max_id=max(region_by_ids))
    _clear_unique_fields_that_are_in_rasp(rasp_region_data)  # this can be done only after mapping
    _process_mapped_regions(mappings, tz_repository)


def _shift_and_hide_regions(regions_not_in_rasp, max_id):
    new_id = max(max_id + 1, MIN_NOT_RASP_REGION_ID)
    for region in regions_not_in_rasp:
        update_spec = {'hidden': True}
        if region.id < MIN_NOT_RASP_REGION_ID:
            update_spec['id'] = new_id
            new_id += 1
        Region.objects.filter(id=region.id).update(**update_spec)
        log.info('Not in rasp %s', _region_to_text(Region.objects.get(id=update_spec.get('id', region.id))))


def _process_mapped_regions(mappings, tz_repository):
    regions_to_update = []
    region_new_titles_to_update = []
    for mapping in mappings:
        rasp_region = mapping.rasp_region
        region = mapping.region
        try:
            timezone = tz_repository.get(rasp_region.TimeZoneId).Code
        except Exception:
            timezone = None
            log.error('Could not find TimeZoneId %s in timezone_repository', rasp_region.TimeZoneId)
        spec = {
            'id': rasp_region.Id,
            '_geo_id': rasp_region.GeoId or None,
            'agent_geo_id': rasp_region.AgentGeoId or None,
            'title': rasp_region.TitleDefault,
            '_kladr_id': rasp_region.KladrId,
            'koatuu': rasp_region.Koatuu or None,
            'country_id': rasp_region.CountryId or None,
            'time_zone': timezone,
            'hidden': rasp_region.IsHidden,
            'disputed_territory': rasp_region.IsDisputedTerritory,
            'title_ru': rasp_region.TitleNominative.Ru,
            'title_en': rasp_region.TitleNominative.En,
            'title_uk': rasp_region.TitleNominative.Uk,
            'title_tr': rasp_region.TitleNominative.Tr,
        }
        if not region:
            region = Region.objects.create(**spec)
            log.info('Created %s', _region_to_text(region))
        elif region.id != rasp_region.Id:
            Region.objects.filter(id=region.id).update(**spec)
            region = Region.objects.get(id=rasp_region.Id)
        else:
            for field, value in six.iteritems(spec):
                setattr(region, field, value)
            regions_to_update.append(region)

        region.new_L_title.ru_nominative = rasp_region.TitleNominative.Ru
        region.new_L_title.en_nominative = rasp_region.TitleNominative.En
        region.new_L_title.uk_nominative = rasp_region.TitleNominative.Uk
        region.new_L_title.tr_nominative = rasp_region.TitleNominative.Tr
        region_new_titles_to_update.append(region.new_L_title)

    bulk_update(regions_to_update, batch_size=1000, update_fields=REGION_UPDATE_FIELDS)
    bulk_update(region_new_titles_to_update, batch_size=1000, update_fields=REGION_NEW_TITLE_UPDATE_FIELDS)


def _clear_unique_fields_that_are_in_rasp(rasp_region_data):
    for field, proto_attr in REGION_UNIQUE_FIELDS_AND_PROTO_ATTRS:
        field_values = {getattr(rasp_region, proto_attr) for rasp_region in rasp_region_data}
        for values in chunker(field_values, 1000):
            Region.objects.filter(**{'{}__in'.format(field): values}).update(**{field: None})


def _region_to_text(region):
    return 'Region id={id}, geo_id={geo_id}, title="{title}"'.format(
        id=region.id,
        geo_id=region._geo_id,
        title=force_text(region.title)
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(log)
    create_current_file_run_log()

    log.info('Start sync www_region')

    try:
        do_sync()
    except Exception:
        log.exception('Error in sync www_region')
        raise
    else:
        log.info('Successfully sync www_region')


def do_sync():
    tz_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_TIMEZONE_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    region_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_REGION_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    sync_region(region_repository, tz_repository)
