# coding: utf-8
from __future__ import unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import argparse
import logging
from collections import namedtuple

import six
from django.conf import settings
from django.db import transaction
from django.db.models import Q
from django.utils.encoding import force_text
from django_bulk_update.helper import bulk_update

from travel.avia.admin.avia_scripts.sync_with_rasp.helpers import has_diff
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.admin.lib.iterators import chunker
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType

log = logging.getLogger(__name__)

MIN_NOT_RASP_SETTLEMENT_ID = 500000
SettlementMapping = namedtuple('SettlementMapping', 'rasp_settlement, settlement')

SETTLEMENT_UNIQ_FIELDS_AND_PROTO_ATTRS = (
    ('sirena_id', 'SirenaId'),
    ('iata', 'Iata'),
    ('koatuu', 'Koatuu'),
)

SETTLEMENT_UPDATE_FIELDS = (
    '_geo_id',
    'time_zone',
    'majority_id',
    'suggest_order',
    'country_id',
    'region_id',
    'title',
    'title_ru',
    'title_ru_preposition_v_vo_na',
    'title_ru_genitive',
    'title_ru_accusative',
    'title_ru_locative',
    'title_uk',
    'title_tr',
    'title_en',
    'abbr_title',
    'abbr_title_ru',
    'abbr_title_en',
    'abbr_title_uk',
    'abbr_title_tr',
    'phone_info',
    'phone_info_short',
    '_kladr_id',
    'sirena_id',
    'iata',
    'koatuu',
    'agent_geo_id',
    'big_city',
    'has_tablo',
    'has_many_airports',
    '_disputed_territory',
    'longitude',
    'latitude',
    'hidden',
)

SETTLEMENT_NEW_TITLE_UPDATE_FIELDS = (
    'ru_nominative',
    'ru_genitive',
    'ru_accusative',
    'ru_locative',
    'en_nominative',
    'uk_nominative',
    'tr_nominative',
)

SETTLEMENT_NEW_ABBR_TITLE_UPDATE_FIELDS = (
    'ru_nominative',
    'en_nominative',
    'uk_nominative',
    'tr_nominative',
)


@transaction.atomic
def sync_settlement(settlement_repository, tz_repository):
    rasp_settlements_data = list(settlement_repository.itervalues())
    settlement_by_ids = {r.id: r for r in Settlement.objects.all().select_related('new_L_title', 'new_L_abbr_title')}
    settlement_by_geo_ids = {r._geo_id: r for r in six.itervalues(settlement_by_ids) if r._geo_id}

    mappings = []
    # set mappings by id
    mapped_settlement_ids = set()
    mapped_rasp_settlement_ids = set()
    for rasp_settlement in rasp_settlements_data:
        settlement = settlement_by_ids.get(rasp_settlement.Id)
        if settlement:
            mappings.append(SettlementMapping(rasp_settlement, settlement))
            mapped_settlement_ids.add(settlement.id)
            mapped_rasp_settlement_ids.add(rasp_settlement.Id)

    # now try to map by geo_id
    for rasp_settlement in rasp_settlements_data:
        if rasp_settlement.Id in mapped_rasp_settlement_ids:
            continue

        settlement = settlement_by_geo_ids.get(rasp_settlement.GeoId)
        if settlement and settlement.id not in mapped_settlement_ids:
            mappings.append(SettlementMapping(rasp_settlement, settlement))
            mapped_settlement_ids.add(settlement.id)
            mapped_rasp_settlement_ids.add(rasp_settlement.Id)

    # other rasp settlements maps to nothing
    for rasp_settlement in rasp_settlements_data:
        if rasp_settlement.Id not in mapped_rasp_settlement_ids:
            mappings.append(SettlementMapping(rasp_settlement, None))

    settlements_not_in_rasp = [s for s_id, s in six.iteritems(settlement_by_ids) if s_id not in mapped_settlement_ids]

    # Need to clean uniq fields first, for example, when rasp swaps geo_id,
    # and it is safe to do so, because we do it in transaction,
    # and we have right values in rasp.
    # Note: SET UNIQUE_CHECKS=0 Wouldn't help
    # https://stackoverflow.com/questions/8410844/mysql-temporarily-suppress-unique-index#comment44742725_24132242
    _clear_geo_ids(mapped_settlement_ids)

    _clear_uniq_fields_that_are_in_rasp(rasp_settlements_data)

    _remove_absolute_geo_id(settlements_not_in_rasp, rasp_settlements_data)
    _shift_and_hide_settlements(settlements_not_in_rasp, max_id=max(settlement_by_ids))
    _process_mapped_settlements(mappings, tz_repository)


def _remove_absolute_geo_id(settlements_not_in_rasp, rasp_settlements_data):
    rasp_geo_ids = {s.GeoId for s in rasp_settlements_data if s.GeoId}
    for s in settlements_not_in_rasp:
        if s._geo_id and s._geo_id in rasp_geo_ids:
            log.info('%s is not in rasp, but with rasp geo_id, so make geo_id=None', _settlement_to_text(s))
            s._geo_id=None
            s.save()


def _clear_uniq_fields_that_are_in_rasp(rasp_settlements_data):
    for field, proto_attr in SETTLEMENT_UNIQ_FIELDS_AND_PROTO_ATTRS:
        field_values = {getattr(rasp_settlement, proto_attr) for rasp_settlement in rasp_settlements_data}
        for chunk in chunker(field_values, 1000):
            filter_condition = Q()
            for value in chunk:
                filter_condition |= Q(**{field: value})
            Settlement.objects.filter(filter_condition).update(**{field: None})


def _shift_and_hide_settlements(settlements_not_in_rasp, max_id):
    new_id = max(max_id + 1, MIN_NOT_RASP_SETTLEMENT_ID)
    for settlement in settlements_not_in_rasp:
        update_spec = {'hidden': True}
        if settlement.id < MIN_NOT_RASP_SETTLEMENT_ID:
            update_spec['id'] = new_id
            new_id += 1
        if has_diff(settlement, update_spec):
            Settlement.objects.filter(id=settlement.id).update(**update_spec)
            actual_id = update_spec.get('id', settlement.id)
            if settlement.id != actual_id:
                settlement = Settlement.objects.select_related('new_L_title', 'new_L_abbr_title').get(id=actual_id)

        log.info('Not in rasp %s', _settlement_to_text(settlement))


def _process_mapped_settlements(mappings, tz_repository):
    settlements_to_update = []
    new_titles_to_update = []
    new_abbr_titles_to_update = []
    for mapping in mappings:
        rasp_settlement = mapping.rasp_settlement
        settlement = mapping.settlement
        try:
            timezone = tz_repository.get(rasp_settlement.TimeZoneId).Code
        except Exception:
            timezone = None
            log.error('Could not find TimeZoneId %s in timezone_repository', rasp_settlement.TimeZoneId)
        spec = {
            'id': rasp_settlement.Id,
            '_geo_id': rasp_settlement.GeoId or None,
            'time_zone': timezone,
            'majority_id': rasp_settlement.Majority,
            'suggest_order': rasp_settlement.SuggestOrder,
            'country_id': rasp_settlement.CountryId or None,
            'region_id': rasp_settlement.RegionId or None,
            'title': rasp_settlement.TitleDefault,
            'title_ru': rasp_settlement.Title.Ru.Nominative,
            'title_ru_preposition_v_vo_na': rasp_settlement.Title.Ru.LocativePreposition,
            'title_ru_genitive': rasp_settlement.Title.Ru.Genitive,
            'title_ru_accusative': rasp_settlement.Title.Ru.Accusative,
            'title_ru_locative': rasp_settlement.Title.Ru.Prepositional,
            'title_uk': rasp_settlement.Title.Uk.Nominative,
            'title_tr': rasp_settlement.Title.Tr.Nominative,
            'title_en': rasp_settlement.Title.En.Nominative,
            'abbr_title': rasp_settlement.AbbrTitleDefault,
            'abbr_title_ru': rasp_settlement.AbbrTitle.Ru,
            'abbr_title_en': rasp_settlement.AbbrTitle.En,
            'abbr_title_uk': rasp_settlement.AbbrTitle.Uk,
            'abbr_title_tr': rasp_settlement.AbbrTitle.Tr,
            'phone_info': rasp_settlement.PhoneInfo,
            'phone_info_short': rasp_settlement.PhoneInfoShort,
            '_kladr_id': rasp_settlement.KladrId or None,
            'sirena_id': rasp_settlement.SirenaId or None,
            'iata': rasp_settlement.Iata or None,
            'koatuu': rasp_settlement.Koatuu or None,
            'agent_geo_id': rasp_settlement.AgentGeoId or None,
            'big_city': rasp_settlement.BigCity,
            'has_tablo': rasp_settlement.HasTablo,
            'has_many_airports': rasp_settlement.HasManyAirports,
            '_disputed_territory': rasp_settlement.IsDisputedTerritory,
            'longitude': rasp_settlement.Longitude,
            'latitude': rasp_settlement.Latitude,
            'hidden': rasp_settlement.IsHidden,
        }
        if not settlement:
            settlement = Settlement.objects.create(**spec)
            log.info('Created %s', _settlement_to_text(settlement))
        elif settlement.id != rasp_settlement.Id:
            Settlement.objects.filter(id=settlement.id).update(**spec)
            settlement = Settlement.objects.select_related('new_L_title', 'new_L_abbr_title').get(
                id=rasp_settlement.Id)
        else:
            for field, value in six.iteritems(spec):
                setattr(settlement, field, value)
            settlements_to_update.append(settlement)

        settlement.new_L_title.ru_nominative = rasp_settlement.Title.Ru.Nominative
        settlement.new_L_title.ru_genitive = rasp_settlement.Title.Ru.Genitive
        settlement.new_L_title.ru_accusative = rasp_settlement.Title.Ru.Accusative
        settlement.new_L_title.ru_locative = rasp_settlement.Title.Ru.Prepositional
        settlement.new_L_title.en_nominative = rasp_settlement.Title.En.Nominative
        settlement.new_L_title.uk_nominative = rasp_settlement.Title.Uk.Nominative
        settlement.new_L_title.tr_nominative = rasp_settlement.Title.Tr.Nominative
        new_titles_to_update.append(settlement.new_L_title)

        settlement.new_L_abbr_title.ru_nominative = rasp_settlement.AbbrTitle.Ru
        settlement.new_L_abbr_title.en_nominative = rasp_settlement.AbbrTitle.En
        settlement.new_L_abbr_title.uk_nominative = rasp_settlement.AbbrTitle.Uk
        settlement.new_L_abbr_title.tr_nominative = rasp_settlement.AbbrTitle.Tr
        new_abbr_titles_to_update.append(settlement.new_L_abbr_title)

    bulk_update(settlements_to_update, batch_size=1000, update_fields=SETTLEMENT_UPDATE_FIELDS)
    bulk_update(new_titles_to_update, batch_size=1000, update_fields=SETTLEMENT_NEW_TITLE_UPDATE_FIELDS)
    bulk_update(new_abbr_titles_to_update, batch_size=1000, update_fields=SETTLEMENT_NEW_ABBR_TITLE_UPDATE_FIELDS)


def _clear_geo_ids(settlement_ids):
    for chunk in chunker(settlement_ids, 1000):
        Settlement.objects.filter(id__in=chunk).update(_geo_id=None)


def _settlement_to_text(settlement):
    return 'Settlement id={id}, geo_id={geo_id}, title="{title}"'.format(
        id=settlement.id,
        geo_id=settlement._geo_id,
        title=force_text(settlement.title),
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--verbose', action='store_true')
    args = parser.parse_args()

    if args.verbose:
        add_stdout_handler(log)
    create_current_file_run_log()

    log.info('Start sync www_settlement')

    try:
        do_sync()
    except Exception:
        log.exception('Error in sync www_settlement')
        raise
    else:
        log.info('Successfully sync www_settlement')


def do_sync():
    tz_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_TIMEZONE_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    settlement_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_SETTLEMENT_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    sync_settlement(settlement_repository, tz_repository)
