# -*- encoding: utf-8 -*-
from __future__ import unicode_literals, print_function, division

import travel.avia.admin.init_project  # noqa

import logging
from copy import copy
from optparse import Option, OptionParser

import six
from django.conf import settings
from django.db import transaction
from django.db.models import Q
from django.utils.encoding import force_text
from django_bulk_update.helper import bulk_update

from travel.avia.library.python.common.models.geo import Station, StationCode, CodeSystem
from travel.avia.admin.lib.iterators import chunker
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType
from travel.proto.dicts.rasp.station_pb2 import TStation

log = logging.getLogger(__name__)

IMPORTANT_CODE_SYSTEMS = [1, 4, 5]
STATION_SELECT_RELATED_FIELDS = ('new_L_popular_title', 'new_L_title', 'new_L_how_to_get_to_city', 'new_L_address')
STATION_UNIQUE_FIELDS_AND_PROTO_GETTERS = (
    ('sirena_id', lambda x: x.StationCodes[1]),
    ('express_id', lambda x: x.StationCodes[2]),
)
USE_DIRECTION_MAPPING = {
    TStation.DIRECTION_DO_NOT_USE_DIR: '',
    TStation.DIRECTION_USE_DIR: 'dir',
    TStation.DIRECTION_USE_SUBDIR: 'subdir',
}

STATION_UPDATE_FIELDS = (
    'address',
    'title',
    'popular_title',
    'majority_id',
    'settlement_id',
    'region_id',
    'country_id',
    'time_zone',
    't_type_id',
    'type_choices',
    'hidden',
    'express_id',
    'sirena_id',
    'time_zone_not_check',
    'site_url',
    'not_generalize',
    'station_type_id',
    'longitude',
    'latitude',
    'map_zoom',
    'use_direction',
    'has_aeroexpress',
    'near_metro',
    'photo',
    'schema_image',
    'panorama_url',
    'show_settlement',
    'tablo_state',
    'tablo_state_prev',
    'fuzzy_only',
    'virtual_end',
    'incomplete_bus_schedule',
    'show_mode',
    'is_fuzzy',
    'is_searchable_to',
    'is_searchable_from',
    'in_station_schedule',
    'in_thread',
    'show_tablo_stat',
    'title_ru_override',
    'title_en_override',
    'title_uk_override',
    'title_tr_override',
    'title_ru_preposition_v_vo_na',
)

STATION_NEW_POPULAR_TITLE_UPDATE_FIELDS = (
    'ru_nominative',
    'en_nominative',
    'uk_nominative',
    'tr_nominative',
)

STATION_NEW_TITLE_UPDATE_FIELDS = (
    'ru_nominative',
    'ru_genitive',
    'ru_accusative',
    'ru_locative',
    'en_nominative',
    'uk_nominative',
    'tr_nominative',
)

STATION_HOW_TO_GET_TO_CITY_UPDATE_FIELDS = (
    'ru',
    'en',
    'uk',
    'tr',
)

STATION_ADDRESS_UPDATE_FIELDS = (
    'ru',
    'en',
    'uk',
    'tr',
)


def main():
    create_current_file_run_log()

    optparser = OptionParser(option_class=Yoption)
    optparser.add_option('-v', '--verbose', action='store_true')
    options, args = optparser.parse_args()

    if options.verbose:
        add_stdout_handler(log)

    log.info('Start')

    try:
        do_sync()
    except Exception:
        log.exception('Error in sync stations.')
        raise


def do_sync():
    station_repository = get_repository(
        ResourceType.TRAVEL_DICT_RASP_STATION_PROD,
        oauth=settings.SANDBOX_OAUTH_TOKEN or None,
    )
    sync_stations(station_repository)


@transaction.atomic()
def sync_stations(station_repository):
    # we do not care about stations without important codes
    rasp_stations_data = [
        station for station in station_repository.itervalues()
        if any(system_id in station.StationCodes for system_id in IMPORTANT_CODE_SYSTEMS)
    ]

    # Need to clean station_codes first, because pair ('system', 'code') is uniq,
    # and it is safe to do so, because we do it in transaction,
    # and we have right values in rasp.
    _clear_station_codes(rasp_stations_data)
    _clear_station_uniq_fields(rasp_stations_data)

    _zero_out_deleted_stations(rasp_stations_data)

    station_by_ids = _get_stations_with_codes_to_update(rasp_stations_data)
    _update_stations(rasp_stations_data, station_by_ids)


def _get_stations_with_codes_to_update(rasp_stations_data):
    """
    Get stations and fill db_codes attribute with codes dict
    """
    station_by_ids = {}
    for chunk in chunker(rasp_stations_data, 1000):
        ids = [rasp_station.Id for rasp_station in chunk]
        station_by_ids.update({
            station.id: station
            for station in Station.objects.filter(id__in=ids).select_related(*STATION_SELECT_RELATED_FIELDS)
        })

    for station in six.itervalues(station_by_ids):
        station.db_codes = {}
    for ids_chunk in chunker((rasp_station.Id for rasp_station in rasp_stations_data), 1000):
        for v in StationCode.objects.filter(station_id__in=ids_chunk).values():
            station_by_ids[v['station_id']].db_codes[v['system_id']] = v['code']

    return station_by_ids


def _update_stations(stations_data, station_by_ids):
    known_code_system_ids = set(CodeSystem.objects.all().values_list('id', flat=True))

    log.info('Number of stations to update %s', len(stations_data))
    stations_to_update = []
    new_popular_titles_to_update = []
    new_titles_to_update = []
    new_how_to_get_to_city_to_update = []
    new_addresses_to_update = []
    station_codes_to_update = []
    station_codes_to_create = []
    for rasp_station in stations_data:
        created = False
        station = station_by_ids.get(rasp_station.Id)
        spec = {
            'id': rasp_station.Id,
            'address': rasp_station.LocalAddress,
            'title': rasp_station.LocalTitle,
            'popular_title': rasp_station.LocalPopularTitle,
            'majority_id': rasp_station.Majority,
            'settlement_id': rasp_station.SettlementId or None,
            'region_id': rasp_station.RegionId or None,
            'country_id': rasp_station.CountryId or None,
            'time_zone': rasp_station.TimeZoneCode,
            't_type_id': rasp_station.TransportType,
            'type_choices': rasp_station.TypeChoices,
            'hidden': rasp_station.IsHidden,
            'express_id': rasp_station.StationCodes[2] or None,
            'sirena_id': rasp_station.StationCodes[1] or None,
            'time_zone_not_check': rasp_station.TimeZoneNotCheck,
            'site_url': rasp_station.SiteUrl,
            'not_generalize': rasp_station.NotGeneralize,
            'station_type_id': rasp_station.Type,
            'longitude': rasp_station.Longitude,
            'latitude': rasp_station.Latitude,
            'map_zoom': rasp_station.MapZoom,
            'use_direction': USE_DIRECTION_MAPPING[rasp_station.UseDirection] or None,
            'has_aeroexpress': rasp_station.HasAeroexpress,
            'near_metro': rasp_station.NearMetro,
            'photo': rasp_station.Photo,
            'schema_image': rasp_station.SchemaImage,
            'panorama_url': rasp_station.PanoramaUrl,
            'show_settlement': rasp_station.ShowSettlement,
            'tablo_state': rasp_station.TabloState,
            'tablo_state_prev': rasp_station.TabloStatePrev,
            'fuzzy_only': rasp_station.FuzzyOnly,
            'virtual_end': rasp_station.VirtualEnd,
            'incomplete_bus_schedule': rasp_station.IncompleteBusSchedule,
            'show_mode': rasp_station.ShowMode,
            'is_fuzzy': rasp_station.IsFuzzy,
            'is_searchable_to': rasp_station.IsSearchableTo,
            'is_searchable_from': rasp_station.IsSearchableFrom,
            'in_station_schedule': rasp_station.InStationSchedule,
            'in_thread': rasp_station.InThread,
            'show_tablo_stat': rasp_station.ShowTabloStat,
            'title_ru_override': rasp_station.ShouldOverrideTitle.Ru,
            'title_en_override': rasp_station.ShouldOverrideTitle.En,
            'title_uk_override': rasp_station.ShouldOverrideTitle.Uk,
            'title_tr_override': rasp_station.ShouldOverrideTitle.Tr,
            'title_ru_preposition_v_vo_na': rasp_station.TitleRuPreposition,
        }
        if not station:
            station = Station.objects.create(**spec)
            station.db_codes = {}
            created = True
        elif station.id != rasp_station.Id:
            Station.objects.filter(id=station.id).update(**spec)
            station = Station.objects.select_related(*STATION_SELECT_RELATED_FIELDS).get(id=rasp_station.Id)
            station.db_codes = {v['system_id']: v['code'] for v in station.code_set.values('system_id', 'code')}
        else:
            for field, value in six.iteritems(spec):
                setattr(station, field, value)
            stations_to_update.append(station)

        station.new_L_popular_title.ru_nominative = rasp_station.PopularTitle.Ru
        station.new_L_popular_title.en_nominative = rasp_station.PopularTitle.En
        station.new_L_popular_title.uk_nominative = rasp_station.PopularTitle.Uk
        station.new_L_popular_title.tr_nominative = rasp_station.PopularTitle.Tr
        new_popular_titles_to_update.append(station.new_L_popular_title)

        station.new_L_title.ru_nominative = rasp_station.Title.Ru
        station.new_L_title.ru_genitive = rasp_station.TitleRuGenitiveCase
        station.new_L_title.ru_accusative = rasp_station.TitleRuAccusativeCase
        station.new_L_title.ru_locative = rasp_station.TitleRuPrepositionalCase
        station.new_L_title.en_nominative = rasp_station.Title.En
        station.new_L_title.uk_nominative = rasp_station.Title.Uk
        station.new_L_title.tr_nominative = rasp_station.Title.Tr
        new_titles_to_update.append(station.new_L_title)

        station.new_L_how_to_get_to_city.ru = rasp_station.HowToGetToCity.Ru
        station.new_L_how_to_get_to_city.en = rasp_station.HowToGetToCity.En
        station.new_L_how_to_get_to_city.uk = rasp_station.HowToGetToCity.Uk
        station.new_L_how_to_get_to_city.tr = rasp_station.HowToGetToCity.Tr
        new_how_to_get_to_city_to_update.append(station.new_L_how_to_get_to_city)

        station.new_L_address.ru = rasp_station.Address.Ru
        station.new_L_address.en = rasp_station.Address.En
        station.new_L_address.uk = rasp_station.Address.Uk
        station.new_L_address.tr = rasp_station.Address.Tr
        new_addresses_to_update.append(station.new_L_address)

        for system_id, code in six.iteritems(rasp_station.StationCodes):
            if system_id not in known_code_system_ids:
                log.warning('Unknown code system %s', system_id)
                continue

            if code:
                station_code = StationCode(station_id=rasp_station.Id, system_id=system_id, code=code)
                if system_id not in station.db_codes:
                    station_codes_to_create.append(station_code)
                else:
                    station_codes_to_update.append(station_code)
                station.db_codes[system_id] = code

        if created:
            log.info('Created %s', _station_to_text(station))

    bulk_update(stations_to_update, batch_size=1000, update_fields=STATION_UPDATE_FIELDS)
    bulk_update(new_titles_to_update, batch_size=10000, update_fields=STATION_NEW_TITLE_UPDATE_FIELDS)
    bulk_update(new_popular_titles_to_update, batch_size=10000, update_fields=STATION_NEW_POPULAR_TITLE_UPDATE_FIELDS)
    bulk_update(new_addresses_to_update, batch_size=10000, update_fields=STATION_ADDRESS_UPDATE_FIELDS)
    bulk_update(new_how_to_get_to_city_to_update, batch_size=10000, update_fields=STATION_HOW_TO_GET_TO_CITY_UPDATE_FIELDS)
    bulk_update(station_codes_to_update, batch_size=10000)
    StationCode.objects.bulk_create(station_codes_to_create, batch_size=10000)


def _zero_out_deleted_stations(stations_data):
    # make stations hidden and delete station codes from www_stationcode
    ids_in_database = set(
        Station.objects.filter(
            Q(hidden=False) | ~Q(sirena_id=None) | ~Q(express_id=None)).values_list(
            'pk', flat=True)
    )
    log.info('%d non-hidden stations in database', len(ids_in_database))
    for datum in stations_data:
        ids_in_database.discard(datum.Id)

    log.info(
        '%d non-hidden stations are present in database, but were not found in rasp dump. They will be hidden',
        len(ids_in_database),
    )
    station_codes_to_delete = StationCode.objects.filter(station_id__in=ids_in_database)
    station_codes_to_delete.delete()
    deleted_stations = Station.objects.filter(pk__in=ids_in_database)
    log.info('Current values:')
    for station in deleted_stations:
        log.info('id: %s, sirena_id: %s, express_id: %s, hidden: %s', station.id, station.sirena_id, station.express_id,
                 station.hidden)
    deleted_stations.update(hidden=True, sirena_id=None, express_id=None)
    log.info('Deleted stations update completed')


def _clear_station_codes(rasp_stations_data):
    for chunk in chunker(rasp_stations_data, 1000):
        filter_condition = Q()
        for rasp_station in chunk:
            for system_id in rasp_station.StationCodes:
                filter_condition |= (Q(system_id=system_id) & Q(code=rasp_station.StationCodes[system_id]))
        StationCode.objects.filter(filter_condition).delete()


def _clear_station_uniq_fields(stations_data):
    for field, proto_getter in STATION_UNIQUE_FIELDS_AND_PROTO_GETTERS:
        field_values = {proto_getter(rasp_station) for rasp_station in stations_data
                        if proto_getter(rasp_station)}
        for chunk in chunker(field_values, 1000):
            Station.objects.filter(**{'{}__in'.format(field): chunk}).update(**{field: None})


def _station_to_text(station):
    db_codes = getattr(station, 'db_codes',
                       {v['system_id']: v['code'] for v in station.code_set.values('system_id', 'code')})
    return 'Station id={s.id}, t_type={s.t_type.code}, iata="{iata}", icao="{icao}", title="{title}"'.format(
        s=station,
        # No db query because CodeSystem is precached in sync_all
        iata=db_codes.get(CodeSystem.objects.get(code='iata').id, ''),
        icao=db_codes.get(CodeSystem.objects.get(code='icao').id, ''),
        title=force_text(station.title)
    )


class Yoption(Option):
    TYPES = Option.TYPES
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
